import java.util.ArrayList;
import java.util.List;
import java.util.Random;

/**
 * The DCJ evolution model.
 */
public class DCJ implements EvolutionModel {
	Random random = new Random();
	private int neighbourGeneration = 3;
	private JoinSolutionsCandidateGenerator JSCGenerator = null;
	
	public DCJ() {
	}
	
	/**
	 * How distant (from the 3 neighbouring nodes) neighbours should be generated (3,2,1,0)
	 */
	public void setNeighbourGeneration(int neighbourGeneration) {
		this.neighbourGeneration = neighbourGeneration;
	}
	
	public void setJSCGenerator(JoinSolutionsCandidateGenerator jSCGenerator) {
		JSCGenerator = jSCGenerator;
	}

	public int distance(Genome genome1, Genome genome2) {
		boolean adj1[] = new boolean[genome1.getGeneCount()*2];
		for (int i = 0; i < adj1.length; i++) {
			adj1[i] = false;
		}
		boolean adj2[] = new boolean[genome2.getGeneCount()*2];
		for (int i = 0; i < adj2.length; i++) {
			adj2[i] = false;
		}
		if (adj1.length!=adj2.length) throw new RuntimeException("Error: Not equal gene content");
		
		int oddPaths = 0;
		int cycles = 0;
		
		for (int i = 0; i < adj1.length; i++) {
			if (adj1[i]==true) continue;
			if (genome1.isIndexTelomere(i)) {
				int componentSize = getComponentSize(genome1, genome2, adj1, adj2, Genome.getExtremityByAdjacencyIndex(i));
				//System.out.println("compsize1: "+componentSize);
				if (componentSize%2==1) oddPaths++;
			}
		}
		for (int i = 0; i < adj2.length; i++) {
			if (adj2[i]==true) continue;
			if (genome2.isIndexTelomere(i)) {
				int componentSize = getComponentSize(genome2, genome1, adj2, adj1, Genome.getExtremityByAdjacencyIndex(i));
				//System.out.println("compsize2: "+componentSize);
				// in fact there should not be odd paths now, only even paths
				if (componentSize%2==1) {
					oddPaths++;
					throw new RuntimeException("Error");
				}
			}
		}
		// cycles
		for (int i = 0; i < adj1.length; i++) {
			if (adj1[i]==true) continue;
				int componentSize = getComponentSize(genome1, genome2, adj1, adj2, Genome.getExtremityByAdjacencyIndex(i));
				//System.out.println("compsize3: "+componentSize);
				cycles++;
		}
		//System.out.println("cycles: "+cycles+", oddPtahs: "+oddPaths);
		return genome1.getGeneCount() - cycles - oddPaths/2;
	}

	/**
	 * Helper function to compute the size of a component of the bipartite DCJ graph.
	 * @param genome1
	 * @param genome2
	 * @param adj1
	 * @param adj2
	 * @param extremity
	 * @return
	 */
	private static int getComponentSize(Genome genome1, Genome genome2, boolean[] adj1, boolean[] adj2, int extremity) {
		int length = 0;
		while (true) {
			int adjacencyIndex = Genome.getAdjacencyIndexByExtremity(extremity);
			
			// we closed a cycle
			if (length%2==0 && adj1[adjacencyIndex]) break;
			if (length%2==1 && adj2[adjacencyIndex]) break;
			
			// mark as visited
			if (length%2==0) {
				adj1[adjacencyIndex] = true;
				adj1[Genome.getAdjacencyIndexByExtremity(genome1.getAdjacentExtremityByExtremity(extremity))] = true;
			}
			if (length%2==1) {
				adj2[adjacencyIndex] = true;
				adj2[Genome.getAdjacencyIndexByExtremity(genome2.getAdjacentExtremityByExtremity(extremity))] = true;
			}
			
			// maybe we are at the end of a path
			if (length>0) {
				if (length%2==0 && genome1.isExtremityTelomere(extremity)) break;
				if (length%2==1 && genome2.isExtremityTelomere(extremity)) break;
			}
			
			// select the next extremity
			if (length%2==0) extremity = genome1.getAdjacentExtremityByIndex(adjacencyIndex);
			if (length%2==1) extremity = genome2.getAdjacentExtremityByIndex(adjacencyIndex);
			
			// increase length
			length++;
		}

		return length;
	}
	
	public void generateCandidates(Phylogeny phylogeny) {
		CandidateGenerator.clearCandidates(phylogeny);
		CandidateGenerator.addSelf(phylogeny);
		CandidateGenerator.addDescendants(phylogeny);
		CandidateGenerator.addParents(phylogeny);
		if (neighbourGeneration==3) this.addCandidatesNeighboursDiff(phylogeny);
		else if (neighbourGeneration<3) this.addCandidatesBetterNeighboursDiff(phylogeny, neighbourGeneration);
		if (JSCGenerator!=null) JSCGenerator.generateCandidates(phylogeny);
	}
	
//	/**
//	 * Generates neighbours and puts them into the candidates list of every internal node in the phylogeny.
//	 * @param phylogeny
//	 */
//	private void addCandidatesNeighbours(Phylogeny phylogeny) {
//		for (PhylogenyNode node : phylogeny.getInternalNodes()) {
//			ArrayList<Genome> candidates = new ArrayList<Genome>();
//			List<Genome> neighbours = generateNeighbours(node.getGenome());
//			candidates.addAll(neighbours);
//			for (Genome candidate : candidates) {
//				node.addCandidate(candidate);
//			}
//		}
//	}
	
	/**
	 * Generates neighbours and their diffs and puts them into the candidates list of every internal node in the phylogeny.
	 * @param phylogeny
	 */
	private void addCandidatesNeighboursDiff(Phylogeny phylogeny) {
		for (PhylogenyNode node : phylogeny.getInternalNodes()) {
			Neighbours neighbours = generateNeighboursDiff(node.getGenome());
			for (int i=0; i<neighbours.size(); i++) {
				node.addCandidate(neighbours.candidate.get(i), neighbours.diffBreak.get(i), neighbours.diffJoin.get(i));
			}
		}
	}

	/**
	 * Generates a list of neighbours. There may be duplicates.
	 * @param genome
	 * @return
	 */
	private List<Genome> generateNeighbours(Genome genome) {
		List<Genome> neighbours = new ArrayList<>();
		for (int i=0; i<genome.getGeneCount()*2; i++) {
			for (int j=0; j<genome.getGeneCount()*2; j++) {
				if (i!=j) {
					Genome neighbour = new Genome(genome);
					dcjOperation(neighbour, i, j);
					neighbours.add(neighbour);
				}
			}
		}
		return neighbours;
	}
	
	private Neighbours generateNeighboursDiff(Genome genome) {
		List<Genome> neighbours = new ArrayList<>();
		List<int[]> breaks = new ArrayList<>();
		List<int[]> joins = new ArrayList<>();
		
		for (int i=0; i<genome.getGeneCount()*2; i++) {
			for (int j=0; j<genome.getGeneCount()*2; j++) {
				if (i==j) continue;
				Genome mutated = new Genome(genome);
				int type = this.dcjOperation(mutated, i, j);
				neighbours.add(mutated);
				
				int extA = Genome.getExtremityByAdjacencyIndex(i);
				int extB = genome.getAdjacentExtremityByIndex(i);
				int extC = Genome.getExtremityByAdjacencyIndex(j);
				int extD = genome.getAdjacentExtremityByIndex(j);
				
				switch (type) {
				case 1:
					// nothing is broken, only join
					breaks.add(new int[0]);
					joins.add(new int[]{extA, extC});
					break;
				case 2:
					breaks.add(new int[]{extA, extB});
					joins.add(new int[]{extA, extC});
					break;
				case 3:
					breaks.add(new int[]{extC, extD});
					joins.add(new int[]{extA, extC});
					break;
				case 4:
					if (extA==extD && extB==extC) {
						// no joins, only break
						breaks.add(new int[]{extA, extB});
						joins.add(new int[0]);
					}
					else {
						breaks.add(new int[]{extA, extB, extC, extD});
						joins.add(new int[]{extA, extD, extB, extC});
					}
					break;
				}
			}
		}
		Neighbours neigh = new Neighbours();
		neigh.candidate = neighbours;
		neigh.diffBreak = breaks;
		neigh.diffJoin = joins;
		return neigh;
	}
	
	/**
	 * Performs a DCJ operation.
	 * @param genome
	 * @param adjacencyIndex1
	 * @param adjacencyIndex2
	 * @return What type of operation was performed
	 */
	public int dcjOperation(Genome genome, int adjacencyIndex1, int adjacencyIndex2) {
		int extA = Genome.getExtremityByAdjacencyIndex(adjacencyIndex1);
		int extB = genome.getAdjacentExtremityByIndex(adjacencyIndex1);
		int extC = Genome.getExtremityByAdjacencyIndex(adjacencyIndex2);
		int extD = genome.getAdjacentExtremityByIndex(adjacencyIndex2);
		// we have (A,B) x (C,D)
		
		if (extA==extB && extC==extD) {			// (X) x (Y) -> (X,Y)
			genome.setAdjacency(extA, extC);
			return 1;
		}
		else if (extA!=extB && extC==extD) {	// (X,Y) x (Z) -> (X,Z) , (Y)
			genome.setAdjacency(extA, extC);
			genome.setAdjacency(extB, extB);
			return 2;
		}
		else if (extA==extB && extC!=extD) {	// (X) x (Y,Z) -> (X,Y) , (Z)
			genome.setAdjacency(extA, extC);
			genome.setAdjacency(extD, extD);
			return 3;
		}
		else {									// (X,Y) x (Z,V) -> (X,V) , (Z,Y)   // (X,Y) x (Y,X) -> (X) , (Y)
			genome.setAdjacency(extA, extD);
			genome.setAdjacency(extC, extB);
			return 4;
		}
	}
	
//	/**
//	 * Generates candidates, adds only those neighbours that increase the distance at most by allowedDistance
//	 * @param phylogeny
//	 * @param allowedDistance
//	 * @see #addCandidatesNeighbours()
//	 */
//	private void addCandidatesBetterNeighbours(Phylogeny phylogeny, int allowedDistance) {
//		for (PhylogenyNode node : phylogeny.getInternalNodes()) {
//			ArrayList<Genome> around = new ArrayList<>();
//			if (node.getParent()!=null) around.add(node.getParent().getGenome());
//			around.add(node.getLeft().getGenome());
//			around.add(node.getRight().getGenome());
//			ArrayList<Genome> candidates = new ArrayList<Genome>();
//			List<Genome> neighbours = generateBetterNeighbours(node.getGenome(), around, allowedDistance);
//			candidates.addAll(neighbours);
//			for (Genome candidate : candidates) {
//				node.addCandidate(candidate);
//			}
//		}
//	}
	
//	private List<Genome> generateBetterNeighbours(Genome genome, ArrayList<Genome> around, int allowedDistance) {
//		int distSum = 0;
//		for (Genome aroundGenome : around) {
//			distSum += distance(genome, aroundGenome);
//		}
//		List<Genome> neighbours = generateNeighbours(genome);
//		ArrayList<Genome> betterNeighbours = new ArrayList<>();
//		
//		for (Genome neighbour : neighbours) {
//			int dist = 0;
//			for (Genome aroundGenome : around) {
//				dist += distance(neighbour, aroundGenome);
//			}
//			if (dist<=distSum+allowedDistance) betterNeighbours.add(neighbour);
//		}
//		return betterNeighbours;
//	}
	
	private void addCandidatesBetterNeighboursDiff(Phylogeny phylogeny, int allowedDistance) {
		for (PhylogenyNode node : phylogeny.getInternalNodes()) {
			ArrayList<Genome> around = new ArrayList<>();
			if (node.getParent()!=null) around.add(node.getParent().getGenome());
			around.add(node.getLeft().getGenome());
			around.add(node.getRight().getGenome());
			Neighbours neighbours = generateBetterNeighboursDiff(node.getGenome(), around, allowedDistance);
			for (int i=0; i<neighbours.size(); i++) {
				node.addCandidate(neighbours.candidate.get(i), neighbours.diffBreak.get(i), neighbours.diffJoin.get(i));
			}
		}
	}
	
	private Neighbours generateBetterNeighboursDiff(Genome genome, ArrayList<Genome> around, int allowedDistance) {
		int distSum = 0;
		for (Genome aroundGenome : around) {
			distSum += distance(genome, aroundGenome);
		}
		Neighbours neighbours = generateNeighboursDiff(genome);
		Neighbours betterNeighbours = new Neighbours();
		
		for (int i=0; i<neighbours.size(); i++) {
			int dist = 0;
			for (Genome aroundGenome : around) {
				dist += distance(neighbours.candidate.get(i), aroundGenome);
			}
			if (dist<=distSum+allowedDistance) {
				betterNeighbours.candidate.add(neighbours.candidate.get(i));
				betterNeighbours.diffBreak.add(neighbours.diffBreak.get(i));
				betterNeighbours.diffJoin.add(neighbours.diffJoin.get(i));
			}
		}
		
		return betterNeighbours;
	}

	public List<Genome> path(Genome genome1, Genome genome2) {
		// TODO
		return null;
	}

	@Override
	public void mutate(Genome genome) {
		int number = genome.getGeneCount()*2;
		int random1 = random.nextInt(number);
		int random2 = (random1 + random.nextInt(number-1)+1)%number;
		dcjOperation(genome, random1, random2);
	}

	@Override
	public EfficientDistance getEfficientDistance(Genome genome1, Genome genome2) {
		return new DcjDistance(genome1, genome2);
	}
	
	class Neighbours {
		public List<Genome> candidate = new ArrayList<>();
		public List<int[]> diffBreak = new ArrayList<>();
		public List<int[]> diffJoin = new ArrayList<>();
		
		public int size() {
			return candidate.size();
		}
	}
}
