import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Random;



/**
 * Genome of an organism. 
 * <p>The genome is internally stored as a set of adjacencies.
 */
public class Genome {
	
	/**
	 * Number of genes.
	 */
	private int geneCount;
	
	/**
	 * Array of adjacencies.
	 * <p>Stores the neighbor of every extremity.
	 * <p>If the extremity is neighboring with itself, then it is a telomere.
	 * <p>The length of the array is 2*geneCount.
	 * 
	 * @see geneCount
	 */
	private int adjacencies[];
	
	/**
	 * Returns the number of genes.
	 */
	public int getGeneCount() {
		return geneCount;
	}
	
	/**
	 * Returns the adjacency index of the gene head.
	 * @param gene
	 * @return adjacency index
	 * @see adjacencies
	 */
	public static int getGeneHeadAdjacencyIndex(int gene) {
		return 2*gene-1;
	}
	
	/**
	 * Returns the adjacency index of the gene tail.
	 * @param gene
	 * @return adjacency index
	 * @see adjacencies
	 */
	public static int getGeneTailAdjacencyIndex(int gene) {
		return 2*gene-2;
	}
	
	/**
	 * Returns the adjacency index of a gene extremity.
	 * @param extremity
	 * @return adjacency index
	 */
	public static int getAdjacencyIndexByExtremity(int extremity) {
		if (extremity<0) return (getGeneTailAdjacencyIndex(-extremity));
		if (extremity>0) return (getGeneHeadAdjacencyIndex(extremity));
		throw new RuntimeException("Genome error");
	}
	
	/**
	 * Returns adjacent extremity based on adjacency index.
	 * @param adjacencyIndex
	 */
	public int getAdjacentExtremityByIndex(int adjacencyIndex) {
		return this.adjacencies[adjacencyIndex];
	}
	
	/**
	 * Returns adjacent extremity based on extremity.
	 * @param extremity
	 */
	public int getAdjacentExtremityByExtremity(int extremity) {
		return this.adjacencies[getAdjacencyIndexByExtremity(extremity)];
	}
	
	/**
	 * Returns which extremity is represented by the adjacency index
	 * @param adjacencyIndex
	 */
	public static int getExtremityByAdjacencyIndex(int adjacencyIndex) {
		int extremity = (adjacencyIndex+2)/2;
		if (adjacencyIndex%2==0) extremity *= -1;
		return extremity;
	}
	
	/**
	 * Sets extremities as adjacent.
	 * @param extremity1
	 * @param extremity2
	 */
	public void setAdjacency(int extremity1, int extremity2) {
		this.adjacencies[getAdjacencyIndexByExtremity(extremity1)]=extremity2;
		this.adjacencies[getAdjacencyIndexByExtremity(extremity2)]=extremity1;
	}
	
	/**
	 * Returns true if the extremity is a telomere. 
	 * @param extremity
	 */
	public boolean isExtremityTelomere(int extremity) {
		int adjacentExtremity = this.getAdjacentExtremityByExtremity(extremity);
		if (adjacentExtremity == extremity) return true;
		else return false;
	}
	
	/**
	 * Returns true if the extremity saved in the adjacency is a telomere.
	 * @param adjacencyIndex
	 * @see #isExtremityTelomere(int)
	 */
	public boolean isIndexTelomere(int adjacencyIndex) {
		return isExtremityTelomere(getExtremityByAdjacencyIndex(adjacencyIndex));
	}

	/**
	 * Creates the genome from the string representation of the genome.
	 * @param genome string
	 * @see parseGenomeString
	 */
	public Genome(String genomeString) {
		parseGenomeString(genomeString);
	}
	
	/**
	 * Cloning constructor.
	 * @param genome
	 */
	public Genome(Genome genome) {
		this.geneCount = genome.geneCount;
		
		this.adjacencies = new int[genome.adjacencies.length];
		System.arraycopy(genome.adjacencies, 0, this.adjacencies, 0, genome.adjacencies.length);
	}

	/**
	 * Parses the genome string
	 * <p>Example string: "-1 2 3 $ 4 $ -5 $ 6 7 @ -8 -9 $"
	 * @param genomeString The string representation of the genome
	 */
	private void parseGenomeString(String genomeString) {
		String[] parts = genomeString.split(" +");
		int numberOfGenes = parts.length - (genomeString.length() - genomeString.replace("$", "").replace("@", "").length());
		this.geneCount = numberOfGenes;
		adjacencies = new int[2*this.geneCount];
		
		int firstGene = 0;
		int lastGene = 0;
		
		for (String part : parts) {
			if (part.equals("$")) {
				// set the first and last genes as telomeres
				this.setAdjacency(-firstGene, -firstGene);
				this.setAdjacency(lastGene, lastGene);
				// reset the first and last genes
				firstGene = 0;
				lastGene = 0;
			}
			else if (part.equals("@")) {
				// we have to connect the first gene with the last gene
				this.setAdjacency(-firstGene, lastGene);
				// reset the first and last genes
				firstGene = 0;
				lastGene = 0;
			}
			else {
				int gene = Integer.parseInt(part);
				if (firstGene == 0) {
					firstGene = gene;
				}
				else if (lastGene !=0) {
					// connect with previous gene
					this.setAdjacency(lastGene, -gene);
				}
				lastGene = gene;
			}
		}
	}
	
	public static String generateRandomGenomeString(int genes) {
		Random random = new Random(); 
		ArrayList<Integer> geneArray = new ArrayList<Integer>();
		for (int i=0; i<genes; i++) {
			geneArray.add(i+1);
		}
		Collections.shuffle(geneArray);
		for (int i = 0; i < geneArray.size(); i++) {
			int orientation = random.nextInt(2)*2-1;  // -1 or 1
			geneArray.set(i, geneArray.get(i)*orientation);
		}
		StringBuilder genomeStringBuilder = new StringBuilder();
		for (int i = 0; i < geneArray.size(); i++) {
			genomeStringBuilder.append(geneArray.get(i)).append(" ");
			
			// Warning: We can generate a genome with multiple chromosomes, but I think Kukos code generated genoms with one chromosome
			String[] chromosome;
			if (i<geneArray.size()-1) chromosome = new String[] {"", "$ ", "@ "};
			else chromosome = new String[] {"$ ", "@ "};
			genomeStringBuilder.append(chromosome[random.nextInt(chromosome.length)]);
		}
		String genomeString = genomeStringBuilder.toString().trim();
		// System.out.println(genomeString);
		return genomeString;
	}
	
	@Override
	public String toString() {
		StringBuilder output = new StringBuilder();
		output.append("Adjacencies:\n");
		for (int i = 0; i < adjacencies.length; i++) {
			output.append("Adjacency "+i + ": " + getExtremityByAdjacencyIndex(i) +" , " + this.getAdjacentExtremityByIndex(i) + "\n");
		}
		return output.toString();
	}
	
	@Override
	public int hashCode() {
		//int hash = this.toString().hashCode();
		int hash = Arrays.hashCode(this.adjacencies);
		return hash;
	}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == null || obj.getClass().equals(Genome.class)) {
			Genome genome = (Genome) obj;
			if (this.geneCount!=genome.geneCount) return false;
			for (int i=0; i<this.geneCount*2; i++) {
				if (this.adjacencies[i]!=genome.adjacencies[i]) return false;
			}
			return true;
		}
		return false;
	}

	public String getStringRepresentation() {
		StringBuilder builder = new StringBuilder();
		boolean[] visited = new boolean[geneCount*2];
		for (int i=0; i<geneCount*2; i++) {
			visited[i] = false;
		}
		for (int i=0; i<geneCount*2; i++) {
			if (!visited[i] && isIndexTelomere(i)) {
				builder.append(getStringRepresentationComponent(i, visited));
				builder.append("$ ");
			}
		}
		for (int i=0; i<geneCount*2; i++) {
			if (!visited[i]) {
				builder.append(getStringRepresentationComponent(i, visited));
				builder.append("@ ");
			}
		}
		String result = builder.toString();
		Genome testGenome = new Genome(result);
		if (!this.equals(testGenome)) throw new RuntimeException("ERROR: Genome.getStringRepresentation()");
		return result;
	}

	private Object getStringRepresentationComponent(int i, boolean[] visited) {
		StringBuilder builder = new StringBuilder();
		int extremity = getExtremityByAdjacencyIndex(i);
		while (visited[getAdjacencyIndexByExtremity(extremity)]==false) {
			visited[getAdjacencyIndexByExtremity(extremity)]=true;
			extremity = -extremity;
			builder.append(extremity + " ");
			visited[getAdjacencyIndexByExtremity(extremity)]=true;
			extremity=getAdjacentExtremityByExtremity(extremity);
		}
		
		return builder.toString();
	}
	
	/**
	 * Returns the number of linear and circular chromosomes
	 * @return number of linear ch., number of circular ch.
	 */
	public int[] getChromosomeCount() {
		boolean visited[] = new boolean[2*getGeneCount()];
		for (int i=0; i<visited.length; i++) visited[i]=false;
		int linear = 0;
		int circular = 0;
		for (int i=0; i<visited.length; i++) {
			if (!visited[i] && isIndexTelomere(i)) {
				markChromosome(i, visited);
				linear++;
			}
		}
		for (int i=0; i<visited.length; i++) {
			if (!visited[i]) {
				markChromosome(i, visited);
				circular++;
			}
		}
		return new int[]{linear,circular};
	}

	private void markChromosome(int index, boolean[] visited) {
		int i = index;
		while (!visited[i]) {
			visited[i] = true;
			visited[getAdjacencyIndexByExtremity(getAdjacentExtremityByIndex(i))] = true;
			i = getAdjacencyIndexByExtremity(-getAdjacentExtremityByIndex(i));
		}
	}
}
