import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;


public class SmallPhylogeny {

	private EvolutionModel evolutionModel;
	private boolean useEfficientDistance = false;
	private Phylogeny initialPhylogeny;
	private Phylogeny bestPhylogeny;
	
	/**
	 * The score of the best tree. The score conteins penalty.
	 */
	private float bestScore;
	
	private Random random;
	private ChromosomePenalty chromosomePenalty;
	private int plateauWalk = 0;
	private TabuSearch tabuSearch;
	
	private String log = "";
	
	public void setUseEfficientDistance(boolean useEfficientDistance) {
		this.useEfficientDistance = useEfficientDistance;
	}
	
	public Phylogeny getBestPhylogeny() {
		return bestPhylogeny;
	}
	
	public float getBestScore() {
		return bestScore;
	}
	
	public void setChromosomePenalty(ChromosomePenalty chromosomePenalty) {
		this.chromosomePenalty = chromosomePenalty;
	}
	
	public void setPlateauWalk(int plateauWalk) {
		this.plateauWalk = plateauWalk;
	}
	
	public void setTabuSearch(TabuSearch tabuSearch) {
		this.tabuSearch = tabuSearch;
	}
	
	public String getLog() {
		return log;
	}
	
	public SmallPhylogeny(EvolutionModel model, Phylogeny phylogeny) {
		this.evolutionModel = model;
		this.initialPhylogeny = new Phylogeny(phylogeny);
		this.bestPhylogeny = null;
		this.bestScore = -1;
		random = new Random();
		chromosomePenalty = new ChromosomePenalty_all();
	}
	
	/**
	 * Randomized search.
	 * @param rounds
	 * @param handler
	 * @return The score of the best tree. The score contains penalty.
	 */
	public float randomizedSearch(int rounds, SmallPhylogenyResultHandler handler) {
		Phylogeny phylogeny = new Phylogeny(initialPhylogeny);
		
		for (int i=0; i<rounds; i++) {
			setLeafGenome(phylogeny);
			randomizePhylogeny(phylogeny);
			float score = findLocalOptimum(phylogeny);
			if (isBetterScore(score)) {
				bestScore = score;
				bestPhylogeny = new Phylogeny(phylogeny);
			}
			if (handler!=null) {
				handler.smallPhylogenyResult(score, phylogeny);
			}
		}
		
		return bestScore;
	}
	
	/**
	 * Simple search.
	 * @param rounds
	 * @param handler
	 * @return The score of the best tree. The score contains penalty.
	 */
	public float simpleSearch(int rounds, SmallPhylogenyResultHandler handler) {
		Phylogeny phylogeny = new Phylogeny(initialPhylogeny);
		
		for (int i=0; i<rounds; i++) {
			setLeafGenome(phylogeny);
			setDescendantAsGenome(phylogeny);
			float score = findLocalOptimum(phylogeny);
			if (isBetterScore(score)) {
				bestScore = score;
				bestPhylogeny = new Phylogeny(phylogeny);
			}
			if (handler!=null) {
				handler.smallPhylogenyResult(score, phylogeny);
			}
		}
		
		return bestScore;
	}

	/**
	 * Generates totally random genomes into the internal nodes of the phylogenetic tree.
	 * <p><strong>Warning:</strong> I think Kukos code generated only one large chromosome. Our code can generate genomes with more than one chromosome.
	 * @see Genome
	 */
	private void randomizePhylogeny(Phylogeny phylogeny) {
		ArrayList<Integer> geneArray = new ArrayList<Integer>();
		for (int i=0; i<phylogeny.getGeneCount(); i++) {
			geneArray.add(i+1);
		}
		for (PhylogenyNode node : phylogeny.getInternalNodes()) {
			Collections.shuffle(geneArray);
			for (int i = 0; i < geneArray.size(); i++) {
				int orientation = random.nextInt(2)*2-1;  // -1 or 1
				geneArray.set(i, geneArray.get(i)*orientation);
			}
			StringBuilder genomeStringBuilder = new StringBuilder();
			for (int i = 0; i < geneArray.size(); i++) {
				genomeStringBuilder.append(geneArray.get(i)).append(" ");
				
				// Warning: We can generate a genome with multiple chromosomes, but I think Kukos code generated genoms with one chromosome
				String[] chromosome;
				if (i<geneArray.size()-1) chromosome = new String[] {"", "$ ", "@ "};
				else chromosome = new String[] {"$ ", "@ "};
				genomeStringBuilder.append(chromosome[random.nextInt(chromosome.length)]);
			}
			String genomeString = genomeStringBuilder.toString().trim();
			// System.out.println(genomeString);
			node.setGenome(new Genome(genomeString));
		}
	}

	/**
	 * A simple helper function for deciding if the score is better than the best score.
	 * @param score
	 * @return
	 */
	private boolean isBetterScore(float score) {
		if (bestScore==-1) return true;
		if (score<bestScore) return true;
		return false;
	}
	
	/**
	 * Finds the local optimum for the given Phylogeny.
	 * @param phylogeny
	 * @return score of the local optimum
	 */
	private float findLocalOptimum(Phylogeny phylogeny) {
		System.out.println("local optimum search started");
		log += "local optimum search started" + System.lineSeparator();
		float newScore = -1;
		float score = 1;
		int plateau = 0;
		do {
			score = newScore;
			evolutionModel.generateCandidates(phylogeny);
			newScore = selectBestCandidates(phylogeny);
			System.out.println("score... " + newScore);
			log += "score... " + newScore + System.lineSeparator();
			if (newScore>=score && score!=-1) plateau++;
			else if (newScore<score) plateau=0;
			if (tabuSearch!=null) tabuSearch.addTabu(phylogeny);
		} while (score==-1 || newScore<score || plateau<plateauWalk);
		return newScore;
	}
	
	/**
	 * Selects the best candidates with dynamic programming.
	 * @param phylogeny
	 * @return score
	 */
	private float selectBestCandidates(Phylogeny phylogeny) {
		if (phylogeny.getRoot().isLeaf()) return 0;
		
		analyzeCandidates(phylogeny.getRoot());
		
		// ok now we look at root and check for which candidates will we get the smallest score
		float bestScore = -1;
		List<Integer> candidatePossibilities = new ArrayList<>();
		for (int i=0; i<phylogeny.getRoot().getCandidateSize(); i++) {
			float actScore = phylogeny.getRoot().getCandidateScore(i);
			if (bestScore==-1 || actScore<bestScore) {
				bestScore = actScore;
				candidatePossibilities.clear();
				candidatePossibilities.add(i);
			}
			else if (bestScore==actScore) {
				candidatePossibilities.add(i);
			}
		}
		// we select the candidate based on the SolutionCount distribution 
		int rootCandidate = selectWithDistribution(candidatePossibilities, phylogeny.getRoot().getSolutionCounts());
		
		// now we can recursively select the best candidates
		selectBestCandidates(phylogeny.getRoot(), rootCandidate);
		
		return bestScore;
	}

	/**
	 * Dynamic programming for computing the best candidates.
	 * <p>Computes the which candidates are the best, and how many solutions would exist 
	 * @param node
	 */
	private void analyzeCandidates(PhylogenyNode node) {
		if (node.isLeaf()) return;
		
		analyzeCandidates(node.getLeft());
		analyzeCandidates(node.getRight());
		
		// set tabu penalty
		if (tabuSearch!=null) {
			for (int i=0; i<node.getCandidateSize(); i++) {
				node.addCandidatePenalty(i, tabuSearch.getPenalty(node.getOrganismName(), node.getCandidate(i)));
			}
		}
		
		// set chromosome penalty
		for (int i=0; i<node.getCandidateSize(); i++) {
			node.addCandidatePenalty(i, chromosomePenalty.calculatePenalty(node.getCandidate(i)));
		}
		
		EfficientDistance efficientDistance = null;
		
		// left subtree scores
		if (useEfficientDistance) efficientDistance = evolutionModel.getEfficientDistance(node.getGenome(), node.getLeft().getGenome());
		for (int i=0; i<node.getCandidateSize(); i++) {
			float minimalScore = -1;
			PhylogenyNode child = node.getLeft();
			for (int j=0; j<child.getCandidateSize(); j++) {
				float score = child.getCandidateScore(j);
				if (efficientDistance!=null && node.isCandidateDiff(i) && child.isCandidateDiff(j)) {
					score += efficientDistance.distance(node.getCandidateDiffBreak(i), node.getCandidateDiffJoin(i), child.getCandidateDiffBreak(j), child.getCandidateDiffJoin(j));
				}
				else score += evolutionModel.distance(node.getCandidate(i), child.getCandidate(j));
				if (minimalScore==-1 || score<minimalScore) {
					minimalScore = score;
					node.deleteLeftSolutions(i);
					node.addLeftSolution(i, j);
				}
				else if (score==minimalScore) {
					node.addLeftSolution(i, j);
				}
			}
			node.setCandidateScoreLeft(i, minimalScore);
		}
		
		// right subtree scores
		if (useEfficientDistance) efficientDistance = evolutionModel.getEfficientDistance(node.getGenome(), node.getRight().getGenome());
		for (int i=0; i<node.getCandidateSize(); i++) {
			float minimalScore = -1;
			PhylogenyNode child = node.getRight();
			for (int j=0; j<child.getCandidateSize(); j++) {
				float score = child.getCandidateScore(j);
				if (efficientDistance!=null && node.isCandidateDiff(i) && child.isCandidateDiff(j)) {
					score += efficientDistance.distance(node.getCandidateDiffBreak(i), node.getCandidateDiffJoin(i), child.getCandidateDiffBreak(j), child.getCandidateDiffJoin(j));
				}
				else score += evolutionModel.distance(node.getCandidate(i), child.getCandidate(j));
				if (minimalScore==-1 || score<minimalScore) {
					minimalScore = score;
					node.deleteRightSolutions(i);
					node.addRightSolution(i, j);
				}
				else if (score==minimalScore) {
					node.addRightSolution(i, j);
				}
			}
			node.setCandidateScoreRight(i, minimalScore);
		}
		
		// update solution counts
		for (int i=0; i<node.getCandidateSize(); i++) {
			int leftCount = 0;
			for (Integer j : node.getLeftSolutions(i)) {
				leftCount += node.getLeft().getSolutionCount(j);
			}
			int rightCount = 0;
			for (Integer j : node.getRightSolutions(i)) {
				rightCount += node.getRight().getSolutionCount(j);
			}
			
			node.setSolutionCount(i, leftCount*rightCount);
		}
	}
	
	private int selectWithDistribution(List<Integer> candidatePossibilities, List<Integer> counts) {
		int sum = 0;
		for (int i=0; i<candidatePossibilities.size(); i++) {
			sum+=counts.get(candidatePossibilities.get(i));
		}
		for (int i=0; i<candidatePossibilities.size(); i++) {
			if (random.nextInt(sum) < counts.get(candidatePossibilities.get(i))) return candidatePossibilities.get(i);
		} 
		return candidatePossibilities.get(candidatePossibilities.size()-1);
	}
	
	private void selectBestCandidates(PhylogenyNode node, int candidate) {
		node.setGenome(new Genome(node.getCandidate(candidate)));
		
		if (!node.isLeaf()) {			
			// now we have to select the candidate in the left and right subtree
			int leftCandidate = selectWithDistribution(node.getLeftSolutions(candidate), node.getLeft().getSolutionCounts());
			int rightCandidate = selectWithDistribution(node.getRightSolutions(candidate), node.getRight().getSolutionCounts());
			
			selectBestCandidates(node.getLeft(), leftCandidate);
			selectBestCandidates(node.getRight(), rightCandidate);
		}
	}
	
	private void setLeafGenome(Phylogeny phylogeny) {
		for (PhylogenyNode leaf : phylogeny.getLeaves()) {
			leaf.setGenome(new Genome(leaf.getCandidate(random.nextInt(leaf.getCandidateSize()))));
		}
	}
	
	private void setDescendantAsGenome(Phylogeny phylogeny) {
		setDescendantAsGenomeRecursive(phylogeny.getRoot());
	}

	private void setDescendantAsGenomeRecursive(PhylogenyNode node) {
		if (node.isLeaf()) return;
		
		setDescendantAsGenomeRecursive(node.getLeft());
		setDescendantAsGenomeRecursive(node.getRight());
		
		ArrayList<Genome> descGenomes = new ArrayList<>();
		descGenomes.add(node.getLeft().getGenome());
		descGenomes.add(node.getRight().getGenome());
		
		Genome selectedGenome = new Genome(descGenomes.get(random.nextInt(descGenomes.size())));
		
		node.setGenome(selectedGenome);
	}
}
