#include "editorscene.h"
#include <QDialog>
#include <assert.h>
#include <iostream>
using namespace std;

EditorScene::EditorScene(QObject *parent)
    : QGraphicsScene(parent) {
    this->line=NULL;
}

void EditorScene::setMode(Mode mode) {
    sceneMode=mode;
};

void EditorScene::deleteItem() {
    if (this->selectedItems().size()) {
        foreach (QGraphicsItem *temp, this->selectedItems()) {
            if (temp!=NULL) {
                if (temp->type()==Edge::Type) {
		  deleteEdge(qgraphicsitem_cast<Edge *>(temp));
                } else {
                    Helix *item = qgraphicsitem_cast<Helix *>(temp);
                    if (item->a->loopEdge!=NULL) { edges.removeAt(edges.indexOf(item->a->loopEdge)); this->removeItem(item->a->loopEdge); delete item->a->loopEdge; }
                    if (item->b->loopEdge!=NULL) { edges.removeAt(edges.indexOf(item->b->loopEdge)); this->removeItem(item->b->loopEdge); delete item->b->loopEdge; }
                    if (item->c->loopEdge!=NULL) { edges.removeAt(edges.indexOf(item->c->loopEdge)); this->removeItem(item->c->loopEdge); delete item->c->loopEdge; }
                    if (item->d->loopEdge!=NULL) { edges.removeAt(edges.indexOf(item->d->loopEdge)); this->removeItem(item->d->loopEdge); delete item->d->loopEdge; }
                    helices.removeAt(helices.indexOf(item));
                    this->removeItem(item);
                    delete item;
                }
            }
        }
    }
}

void EditorScene::deleteEdge(Edge *edge) {
  edges.removeAt(edges.indexOf(edge));
  Node *first=NULL,*second=NULL;
  if (edge->source->small()) {
    if (edge->source->next(edge)==NULL) {
      first=edge->source;
    }
  }
  if (edge->target->small()) {
    if (edge->target->next(edge)==NULL) {
      second=edge->target;
    }
  }
  if (first!=NULL) nodes.removeAt(nodes.indexOf(first));
  if (second!=NULL) nodes.removeAt(nodes.indexOf(second));
  this->removeItem(edge);
  delete edge;  
}

void EditorScene::wheelEvent(QGraphicsSceneWheelEvent *event) {
    if (sceneMode==Connect) {
      // Originally connect node wheel action meant resizing helices
      // Now it is accomplished by editing their pattern
      //        if (this->selectedItems().size()) {
      //      QGraphicsItem *temp=this->selectedItems().first();
      //      if (temp->type()!=Edge::Type) {
      //          Helix *item = qgraphicsitem_cast<Helix *>(temp);
      //          item->resize(item->length+event->delta()/15);
      //      }
      //  }
    } else if (sceneMode==Move) {
        if (this->selectedItems().size()) {
            QGraphicsItem *temp=this->selectedItems().first();
            if (temp->type()!=Edge::Type) {
                Helix *item = qgraphicsitem_cast<Helix *>(temp);
                item->setRotation(item->rotation()+event->delta()/50);
                if (item->a->loopEdge!=NULL) item->a->loopEdge->updatePosition();
                if (item->b->loopEdge!=NULL) item->b->loopEdge->updatePosition();
                if (item->c->loopEdge!=NULL) item->c->loopEdge->updatePosition();
                if (item->d->loopEdge!=NULL) item->d->loopEdge->updatePosition();

            }
        }
    } else {
        QGraphicsScene::wheelEvent(event);
    }
}


void EditorScene::mousePressEvent(QGraphicsSceneMouseEvent *mouseEvent) {
    if (mouseEvent->button() != Qt::LeftButton)
        return;
    switch(sceneMode) {
    case Move:
        break;
    case Connect:
        line = new QGraphicsLineItem(QLineF(mouseEvent->scenePos(),
                                      mouseEvent->scenePos()));
        addItem(line);

        break;
    case Insert:
        line = new QGraphicsLineItem(QLineF(mouseEvent->scenePos(),
                                      mouseEvent->scenePos()));
        addItem(line);
        break;

    }
    QGraphicsScene::mousePressEvent(mouseEvent);
}

void EditorScene::mouseMoveEvent(QGraphicsSceneMouseEvent *mouseEvent) {
    if ((sceneMode == Connect || sceneMode == Insert) && line != NULL) {
         QLineF newLine(line->line().p1(), mouseEvent->scenePos());
         line->setLine(newLine);
     } else if (sceneMode == Move) {
         QGraphicsScene::mouseMoveEvent(mouseEvent);
     }
}

bool EditorScene::checkCycles(Edge *edge) {
  Node *start = edge->source;
  Node *node = start;
  int count = 0;
  int n = nodes.size() + 4*helices.size();
  while(node && edge) {
    Node *nnode = edge->next(node);
    assert(nnode);
    Edge *nedge = nnode->next(edge);
    node = nnode;
    edge = nedge;
    count++;
    if(count > n+1) { return false; }
  }
  return true;
}

bool EditorScene::checkStrand(Edge *edge, int direction) {
  // a depth-first search moving along backbone 
  // as well as opposite edges in a helix
 
  QMap <Edge *, int> dir; // direction of edges visited so far  
  QList <Edge *> stack;  // stack of edges to be processed

  stack.push_back(edge);
  dir[edge] = direction;

  while(! stack.isEmpty()) {
    Edge *curEdge = stack.takeLast();
    int curDir = dir[curEdge];

    // check that the current edge is correct 
    // if it is adjacent to 3' or 5' end
    if((curEdge->source->nodeType=='5' && curDir==-1)
       || (curEdge->source->nodeType=='3' && curDir==1)
       || (curEdge->target->nodeType=='5' && curDir==1)
       || (curEdge->target->nodeType=='3' && curDir==-1)) {
      return false;
    }
    
    QList <Edge *> procEdges;
    QList <int> procDir;

    // try edge following after the target vertex
    Edge *nextEdge = curEdge->target->next(curEdge);
    if(nextEdge) {
      int nextDir = curDir;
      if(curEdge->target != nextEdge->source) {  // edges not directed in the same way
	assert(curEdge->target == nextEdge->target);
	nextDir *= -1;
      }
      procEdges.push_back(nextEdge);
      procDir.push_back(nextDir);
    }

    // try edge following before the source vertex
    nextEdge = curEdge->source->next(curEdge);
    if(nextEdge) {
      int nextDir = curDir;
      if(curEdge->source != nextEdge->target) {  // edges not directed in the same way
	assert(curEdge->source == nextEdge->source);
	nextDir *= -1;
      }
      procEdges.push_back(nextEdge);
      procDir.push_back(nextDir);
    }

    // in a helix try edge on the opposite side
    if(curEdge->myHelix) {
      nextEdge = curEdge->myHelix->opposite(curEdge);
      assert(nextEdge);
      assert(curEdge->myHelix->opposite(curEdge->source)==
	     nextEdge->source);
      procEdges.push_back(nextEdge);
      procDir.push_back(-curDir);
    }

    for(int i=0; i<(int)procEdges.size(); i++) {
      Edge *nextEdge = procEdges[i];
      int nextDir = procDir[i];

      if(dir.contains(nextEdge)) { // this edge already seen
	if(dir[nextEdge]!=nextDir) { return false; }
      }
      else { // new edge
	dir[nextEdge] = nextDir;
	stack.push_back(nextEdge);
      }
    } // for i
  } // while stack

  return true;
}

void EditorScene::mouseReleaseEvent(QGraphicsSceneMouseEvent *mouseEvent) {
    if (line != NULL && sceneMode == Connect) {
         QList<QGraphicsItem *> startItems = items(line->line().p1());
         if (startItems.count() && startItems.first() == line)
             startItems.removeFirst();
         QList<QGraphicsItem *> endItems = items(line->line().p2());
         if (endItems.count() && endItems.first() == line)
             endItems.removeFirst();
         removeItem(line);
         delete line;

         if (startItems.count() > 0 && endItems.count() > 0 &&
             startItems.first() != endItems.first()) {
             Node *start =
                 qgraphicsitem_cast<Node *>(startItems.first());
             Node *end =
                 qgraphicsitem_cast<Node *>(endItems.first());

             if (start->small() && start->helixEdge==NULL) {
                 start->helixEdge=start->loopEdge;
                 start->loopEdge=NULL;
             }
             if (end->small() && end->helixEdge==NULL) {
                 end->helixEdge=end->loopEdge;
                 end->loopEdge=NULL;
             }
             if (start->loopEdge==NULL && end->loopEdge==NULL) {
	       
	       Edge *newEdge = new Edge(start, end, 0, this);
	       edges.insert(edges.size(),newEdge);
	       start->loopEdge=newEdge;
	       end->loopEdge=newEdge;
	       // make all loops empty by default
	       newEdge->loop="e";
	       newEdge->updatePosition();

	       // check if new edge creates a backbone cycle
	       bool ok = checkCycles(newEdge);
	       QString msg="";
	       if(!ok) { 
		 msg = "Wrong structure: Cycle created."; 
	       }
	       else {
		 // Cycles not created, detect incorrect strand
		 ok = checkStrand(newEdge, +1) || checkStrand(newEdge, -1);
		 if(!ok) { 
		   msg = "Wrong structure: Incorrect helix orientation.";
		 }
	       }
	       if(!ok) {
		 // any errors: delete created edge
		 line = NULL;
		 QGraphicsScene::mouseReleaseEvent(mouseEvent);
		 deleteEdge(newEdge);
		 QMessageBox::warning(0, "Cannot connect", msg);
		 return;
	       }
             }
         }
     } else if (line != NULL && sceneMode == Insert) {
         Helix *helix=new Helix(this,line->line().length());
         //this->addItem(helix);
	 //Brona: moved this to Helix constructor to avoid error messages
         // when helix's children are inserted to the scene before parent
         // It would be better to either let all elements insert themselves
         // or not to use scene for the children and insert the whole thing now
         helices.insert(helices.size(),helix);
         helix->setRotation(line->line().angleTo(QLineF(0,0,0,1)));
         helix->setPos(line->line().pointAt(0.5));
         removeItem(line);
         delete line;
     } else if (sceneMode == Insert) {
      Node *node = new Node('s');
      this->addItem(node);
      node->setPos(mouseEvent->lastScenePos());
      node->setFlag(QGraphicsItem::ItemIsMovable,true);
      nodes.insert(nodes.size(),node);
     }
     line = NULL;
     QGraphicsScene::mouseReleaseEvent(mouseEvent);
}
