package graphgui;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Scanner;
import javafx.application.Platform;
import javafx.event.ActionEvent;
import javafx.fxml.FXML;
import javafx.scene.control.Alert;
import javafx.scene.control.Alert.AlertType;
import javafx.scene.control.Button;
import javafx.scene.control.ToggleButton;
import javafx.scene.control.ToggleGroup;
import javafx.scene.control.ComboBox;
import javafx.scene.control.Label;
import javafx.scene.control.Menu;
import javafx.scene.control.MenuBar;
import javafx.scene.control.MenuItem;
import javafx.scene.control.TextArea;
import javafx.scene.control.TextField;
import javafx.scene.input.KeyCode;
import javafx.scene.input.KeyCodeCombination;
import javafx.scene.input.KeyCombination;
import javafx.scene.input.KeyEvent;
import javafx.scene.layout.AnchorPane;
import javafx.scene.layout.BorderPane;
import javafx.stage.Modality;
import javafx.stage.FileChooser;
import javafx.stage.FileChooser.ExtensionFilter;

/** Trieda pre ovládanie grafického rozhrania */
public class Controller implements GraphImplementation.GraphObserver {

    public GraphPane gpGraph;
    public AnchorPane apRight;
    public AnchorPane apLeft;
    @FXML
    public ToggleButton btnAdd;
    public ToggleButton btnSelect;
    public ToggleButton btnDelete;
    public ToggleGroup toggleGroupButtons;
    @FXML
    public Button btnChangeVertexValues;
    @FXML
    public Button btnChangeEdgeValues;
    public TextField tfCommandLine;
    public BorderPane root;
    public AnchorPane apMain;
    public Label labelVertex;
    public TextField tfVertexNumber;
    public TextField tfVertexValue;
    public ComboBox<String> cbVertexColor;
    public ComboBox<String> cbEdgeColor;
    @FXML
    public TextField tfFromVertex;
    @FXML
    public TextField tfToVertex;
    public TextField tfEdgeValue;
    public MenuBar menuBar;
    public Menu menuFile;
    public Menu menuEdit;
    public Menu menuHelp;
    public MenuItem menuItemCloseProgram;
    public MenuItem menuItemLoad;
    public MenuItem menuItemSave;
    public MenuItem menuItemHelp;
    @FXML
    public TextArea taConsole;


    public static final ArrayList<String> VERTEX_COLORS
        = new ArrayList<>(Arrays.asList("white", "gray", "orange", "lightblue", "yellow"));
    public static final ArrayList<String> EDGE_COLORS
        = new ArrayList<>(Arrays.asList("black", "white", "green", "orange", "blue", "yellow"));

    @FXML
    private void initialize() {
        this.btnAdd.setSelected(true);
        this.cbVertexColor.getItems().addAll(VERTEX_COLORS);
        this.cbEdgeColor.getItems().addAll(EDGE_COLORS);
        fillVertexFields();
        fillEdgeFields();
        State.getGraph().addObserver(this);

        this.menuItemSave.setAccelerator(new KeyCodeCombination(
                                             KeyCode.S,
                                             KeyCombination.CONTROL_DOWN
                                         ));
        this.menuItemLoad.setAccelerator(new KeyCodeCombination(
                                             KeyCode.O,
                                             KeyCombination.CONTROL_DOWN
                                         ));
        this.menuItemCloseProgram.setAccelerator(new KeyCodeCombination(
                    KeyCode.Q,
                    KeyCombination.CONTROL_DOWN
                ));
        this.menuItemHelp.setAccelerator(new KeyCodeCombination(
                                             KeyCode.H,
                                             KeyCombination.CONTROL_DOWN
                                         ));

        this.gpGraph.init(this);

        Platform.runLater(() -> {
            this.tfCommandLine.requestFocus();
        });
    }

    @FXML
    protected void changeMode(ActionEvent event) {
        this.gpGraph.updateLine(null);

        Object source = event.getSource();
        if (source == this.btnAdd) {
            State.setMode(State.GraphMode.ADD);
        } else if (source == this.btnSelect) {
            State.setMode(State.GraphMode.SELECT);
        } else if (source == this.btnDelete) {
            State.setMode(State.GraphMode.DELETE);
        }
    }

    /**
     * Ukonci program.
     * @param actionEvent event
     */
    public void closeProgramHandler(ActionEvent actionEvent) {
        System.exit(0);
    }

    /**
     * Otvori dialog na ulozenie grafu.
     * @param actionEvent event
     */
    public void saveGraphHandler(ActionEvent actionEvent) {
        FileChooser chooser = new FileChooser();
        chooser.setTitle("Save graph as");
        chooser.getExtensionFilters().clear();
        chooser.getExtensionFilters().add(new ExtensionFilter(".graph file",
                                          "*.graph"));
        File file = chooser.showSaveDialog(null);
        if (file != null) {
            if (!file.getName().endsWith(".graph")) {
                file = new File(file.getPath() + ".graph");
            }
            String command = String.format("save %s", file);
            runCommand(command, true);
        }
    }

    public String runControllerCommand(String command) {
        if (command.length() == 0 || command.charAt(0) == '#') {
            return "";
        }
        Scanner s = new Scanner(command);
        if (!s.hasNext()) { // prazdny prikaz
            return "";
        }

        String commandName = s.next();
        if (commandName.equals("help")) {
            this.displayHelpAlert();
            return "help displayed";
        }

        if (commandName.equals("quit")) {
            this.closeProgramHandler(new ActionEvent());
            return "quit program failed";
        }

        if (commandName.equals("algorithm")) {
            String result = this.runGraphAlgorithm();
            return "algorithm result: " + result;
        }

        if (commandName.equals("editor")) {
            this.runEditor();
            return "editor executed";
        }

        // unknown command, pass to graph
        return State.getGraph().runCommand(command);
    }

    public String runCommand(String command, boolean showAlert) {
        String result = "";
        try {
            result = runControllerCommand(command);
        } catch (Exception e) {
            result = e.getMessage();
            if (result == null) {
                result = "unspecified error";
            }
            if (showAlert) {
                Alert error = new Alert(AlertType.ERROR, result);
                error.setHeaderText(null);
                error.showAndWait();
            }
        }
        if (!result.equals("")) {
            result = command + "\n# " + result;
        }
        if (!result.equals("")) {
            this.taConsole.appendText(result + "\n");
        }
        return result;
    }

    /**
     * Zobrazí alert s popisom všetkých možných príkazov.
     */
    @SuppressWarnings("checkstyle:LineLength")
    public void displayHelpAlert() {
        Alert alert = new Alert(AlertType.INFORMATION);
        alert.setTitle("Supported Commands");
        alert.setHeaderText(null);
        alert.setContentText(null);
        alert.initModality(Modality.NONE);

        TextArea textArea = new TextArea();
        textArea.setEditable(false);
        textArea.setWrapText(true);
        textArea.setMaxWidth(Double.MAX_VALUE);
        textArea.setMaxHeight(Double.MAX_VALUE);
        alert.getDialogPane().setContent(textArea);
        alert.getDialogPane().setMinHeight(550);
        alert.getDialogPane().setMinWidth(750);

        textArea.appendText("Available commands:\n");
        textArea.appendText("parts in [] are optional\n\n");
        textArea.appendText("save string\n");
        textArea.appendText("open string\n\n");
        textArea.appendText("add vertex [int] [x double] [y double] [color string] [value int]\n");
        textArea.appendText("add edge [int] int [color string] [value int]\n");
        textArea.appendText("update vertex [int] [x double] [y double] [color string] [value int]\n");
        textArea.appendText("update edge [int] [int] [color string] [value int]\n");
        textArea.appendText("remove vertex [int]\n");
        textArea.appendText("remove edge [int] [int]\n\n");

        textArea.appendText("select vertex [int]\n");
        textArea.appendText("select edge [int] int");
        textArea.appendText("deselect vertex\n");
        textArea.appendText("deselect edge\n");
        textArea.appendText("deselect\n\n");

        textArea.appendText("editor\n");
        textArea.appendText("algorithm\n");
        textArea.appendText("quit\n");
        textArea.appendText("help\n\n");

        textArea.appendText("Authors: J. Katreniaková, B. Brejová, S. Gurský, J. Šimo, E. Tesař\n");
        alert.show();
    }

    /**
     * Otvorí dialóg na načítanie grafu.
     * @param actionEvent event
     */
    public void loadGraphHandler(ActionEvent actionEvent) {
        FileChooser chooser = new FileChooser();
        chooser.setTitle("Otvorit graf");
        chooser.getExtensionFilters().clear();
        chooser.getExtensionFilters().add(new ExtensionFilter(".graph file",
                                          "*.graph"));
        File file = chooser.showOpenDialog(null);
        if (file != null) {
            String command = String.format("open %s", file);
            runCommand(command, true);
        }
    }

    /**
     * Zmení stav prvkov o vrchole v pravom paneli.
     * @param b nový stav podaní .setDisable
     */
    @FXML
    public void setDisableVertexValueFields(boolean b) {
        this.btnChangeVertexValues.setDisable(b);
        this.tfVertexValue.setDisable(b);
        this.cbVertexColor.setDisable(b);
    }

    /**
     * Zmení stav prvkov o hrane v pravom paneli.
     * @param b nový stav podaní .setDisable
     */
    @FXML
    public void setDisableEdgeValueFields(boolean b) {
        this.btnChangeEdgeValues.setDisable(b);
        this.tfEdgeValue.setDisable(b);
        this.cbEdgeColor.setDisable(b);
    }

    /**
     * Vyplní prvky o vrchole informáciami vybraného vrcholu.
     */
    public void fillVertexFields() {
        if (State.getGraph().getSelectedVertex() == null) {
            this.tfVertexNumber.setText("N/A");
            this.tfVertexValue.setText("N/A");
            this.setDisableVertexValueFields(true);
            return;
        }
        Vertex v = State.getGraph().getSelectedVertex();
        this.tfVertexNumber.setText(Integer.toString(v.getId()));
        this.tfVertexValue.setText(Integer.toString(v.getValue()));
        this.cbVertexColor.getSelectionModel().select(v.getColorName());
        this.setDisableVertexValueFields(false);
    }

    /**
     * Vyplní prvky o hrane s informáciami o vybranej hrane.
     */
    public void fillEdgeFields() {
        if (State.getGraph().getSelectedEdge() == null) {
            this.tfFromVertex.setText("N/A");
            this.tfToVertex.setText("N/A");
            this.tfEdgeValue.setText("N/A");
            this.setDisableEdgeValueFields(true);
            return;
        }

        Edge e = State.getGraph().getSelectedEdge();
        this.tfFromVertex.setText(Integer.toString(e.getFirstId()));
        this.tfToVertex.setText(Integer.toString(e.getSecondId()));
        this.cbEdgeColor.getSelectionModel().select(e.getColorName());
        this.tfEdgeValue.setText(Integer.toString(e.getValue()));
        this.setDisableEdgeValueFields(false);
    }

    /**
     * Zoberie nové informácie o vrchole z pravého panelu a nastaví ich
     * vybratému vrcholu.
     */
    @FXML
    public void updateVertexValues() {
        if (State.getGraph().getSelectedVertex() == null) {
            return;
        }
        String color
            = (String)this.cbVertexColor.getSelectionModel().getSelectedItem();
        String command
            = String.format("update vertex value %s color %s",
                            this.tfVertexValue.getText().trim(),
                            color);

        runCommand(command, true);
    }

    /**
     * Zoberie nové informácie o hrane z pravého panelu a nastaví ich
     * vybratej hrane.
     */
    @FXML
    public void updateEdgeValues() {
        if (State.getGraph().getSelectedEdge() == null) {
            return;
        }
        String color
            = (String)this.cbEdgeColor.getSelectionModel().getSelectedItem();
        String command
            = String.format("update edge value %s color %s",
                            this.tfEdgeValue.getText().trim(),
                            color);
        runCommand(command, true);
    }

    @FXML
    protected void commandLineTyped(KeyEvent event) {
        if (event.getCode() == KeyCode.ENTER) {
            String command = tfCommandLine.getText();
            this.tfCommandLine.clear();
            runCommand(command, false);
        }
    }

    /**
     * Spustí GraphAlgorithm na graph
     * Vráti jeho výsledok.
     * @return String výsledok behu GraphAlgorithm()
     */
    public String runGraphAlgorithm() {
        GraphAlgorithm ga
            = new GraphAlgorithm(State.getGraph());
        String result = ga.performAlgorithm();
        if (result != null) {
            return result;
        }
        return "Result was null";
    }


    /**
    * Spusti GraphAlgorithm() a jeho vysledok zobrazi alertom.
    */
    public void btnGraphAlgorithmAction() {
        String result = runGraphAlgorithm();
        Alert alert = new Alert(Alert.AlertType.INFORMATION);
        alert.setTitle("Graph Algorithm");
        alert.setHeaderText(null);
        alert.setResizable(false);
        alert.setContentText(result);
        alert.showAndWait();
    }

    public void runEditor() {
        try {
            Editor e = new Editor((Graph)State.getGraph());
            e.edit();
        } catch (EditorException e) {
            Alert alert = new Alert(Alert.AlertType.INFORMATION);
            alert.setTitle("Graph Editor");
            alert.setHeaderText(null);
            alert.setResizable(false);
            alert.setContentText(e.getMessage());
            alert.showAndWait();
        }
    }

    @Override
    public void vertexAdded(Vertex vertex) {
        fillVertexFields();
    }
    @Override
    public void edgeAdded(Edge edge) {
        fillEdgeFields();
    }
    @Override
    public void vertexRemoved(Vertex vertex) {
        fillVertexFields();
    }
    @Override
    public void edgeRemoved(Edge edge) {
        fillEdgeFields();
    }
    @Override
    public void vertexChanged(Vertex vertex) {
        fillVertexFields();
    }
    @Override
    public void edgeChanged(Edge edge) {
        fillEdgeFields();
    }
    @Override
    public void vertexDeselected(Vertex vertex) {
        fillVertexFields();
    }
    @Override
    public void vertexSelected(Vertex vertex) {
        fillVertexFields();
    }
    @Override
    public void edgeDeselected(Edge edge) {
        fillEdgeFields();
    }
    @Override
    public void edgeSelected(Edge edge) {
        fillEdgeFields();
    }
}
