package graphgui;

import javafx.scene.paint.Color;

/**
 * Trieda hrán grafu, implementácia rozhrania Edge.
 * V svojej časti programu pristupujte ku grafu len cez
 * rozhrania Graph, Vertex a Edge.
 */
public final class EdgeImplementation implements Edge {

    public static final int DEFAULT_VALUE = 1;
    private static final String DEFAULT_COLOR_NAME = "black";

    private final GraphImplementation graph;
    private final VertexImplementation first;
    private final VertexImplementation second;
    private int value;
    private String colorName = DEFAULT_COLOR_NAME;

    @Override
    public VertexImplementation getFirst() {
        return first;
    }

    @Override
    public int getFirstId() {
        return getFirst().getId();
    }

    @Override
    public VertexImplementation getSecond() {
        return second;
    }

    @Override
    public int getSecondId() {
        return getSecond().getId();
    }

    @Override
    public boolean isIncident(Vertex vertex) {
        return vertex == first || vertex == second;
    }

    @Override
    public boolean isIncident(int id) {
        return id == first.getId() || id == second.getId();
    }

    @Override
    public Vertex getOtherEnd(Vertex vertex) {
        if (! isIncident(vertex)) {
            throw new IllegalArgumentException("bad vertex");
        }
        if (vertex == first) return second;
        else return first;
    }

    @Override
    public int getOtherEnd(int id) {
        if (! isIncident(id)) {
            throw new IllegalArgumentException("bad vertex id");
        }
        if (id == first.getId()) return second.getId();
        else return first.getId();
    }


    public EdgeImplementation(GraphImplementation graph,
                              VertexImplementation from,
                              VertexImplementation to) {
        this.graph = graph;
        if (from.getId() < to.getId()) {
            this.first = from;
            this.second = to;
        } else {
            this.first = to;
            this.second = from;
        }
        this.value = DEFAULT_VALUE;
        this.colorName = DEFAULT_COLOR_NAME;
    }

    @Override
    public String toString() {
        Integer ifrom = getFirstId();
        Integer ito = getSecondId();
        return String.format("(%s,%s) value %d colorName %s", ifrom, ito, value, colorName);
    }

    @Override
    public void setValue(int value) {
        if (value != this.value) {
            this.value = value;
            graph.graphEdgeChanged(this);
        }
    }

    public static String colorNameFromRgb(int red, int green, int blue)
    throws IllegalArgumentException {
        if (red < 0 || red > 255 || green < 0 || green > 255 || blue < 0 || blue > 255) {
            throw new IllegalArgumentException
            (String.format("bad color values %d %d %d", red, green, blue));
        }
        return String.format("#%02X%02X%02X", red, green, blue);
    }

    @Override
    public void setColorRgb(int red, int green, int blue)
    throws IllegalArgumentException {
        this.setColorName(colorNameFromRgb(red, green, blue));
    }

    public static void verifyColorName(String colorName) {
        try {
            Color c = Color.web(colorName);
        } catch (Exception e) {
            throw new IllegalArgumentException("Bad color name " + colorName);
        }
    }


    @Override
    public void setColorName(String colorName) {
        if (!this.colorName.equals(colorName)) {
            verifyColorName(colorName);
            this.colorName = colorName;
            graph.graphEdgeChanged(this);
        }
    }

    @Override
    public String getColorName() {
        return colorName;
    }

    @Override
    public int getValue() {
        return value;
    }
}
