package graphgui;

import java.io.PrintStream;
import java.util.Scanner;
import java.util.Collection;
import java.util.InputMismatchException;

/**
 * Rozhranie pre prácu s neorientovaným grafom.
 * Umožňuje pridávať a mazať vrcholy aj hrany.
 * Vrcholy aj hrany sú objekty typu Vertex a Edge.
 * Vrcholy sú číslované 0,...,N-1, kde N je počet vrcholov,
 * a v algoritmoch je možné ku nim pristupovať buď cez tieto čísla (id)
 * alebo cez samotné objekty typu Vertex.
 * Pri mazaní vrcholov sa id ostatných vrcholov môžu meniť.
 * Vrcholy aj hrany si pamätajú hodnotu (value) typu int a farbu
 * (color) typu String.  Vrcholy majú uložené aj údaje o svojich
 * súradniciach a veľkosti.
 * V grafe sú neorientované hrany, pričom ako prvý vrchol považujeme
 * ten s menším poradovým číslom.
 * V grafe tiež niektorý vrchol môže byť označený (selected) a podobne
 * môžeme mať aj označenú hranu.
 */
public interface Graph {

    /**
     * Vráti počet vrcholov grafu.
     */
    public int getNumberOfVertices();

    /**
     * Vráti skupinu všetkých vrcholov grafu.
     */
    public Collection<Vertex> getVertices();

    /**
     * Vráti počet hrán grafu.
     */
    public int getNumberOfEdges();

    /**
     * Vráti skupinu všetkých hrán grafu.
     */
    public Collection<Edge> getEdges();

    /**
     * Vráti vrchol s požadovaným id.
     * @param id id požadovaného vrcholu (číslo z intervalu 0..N-1)
     * @throws IllegalArgumentException ak taký vrchol neexistuje
     */
    public Vertex getVertex(int id) throws IllegalArgumentException;

    /**
     * Vráti skupinu vrcholov susediacich s daným vrcholom.
     * @param vertex Vrchol, ktorého susedov požadujeme.
     * @throws IllegalArgumentException ak taký vrchol neexistuje
     */
    public Collection<Vertex> adjVertices(Vertex vertex)
    throws IllegalArgumentException;

    /**
     * Vráti skupinu identifikátorov vrcholov susediacich s vrcholom s daným id.
     * @param id id vrchola, ktorého susedov požadujeme
     * @throws IllegalArgumentException ak taký vrchol neexistuje
     */
    public Collection<Integer> adjVertexIds(int id)
    throws IllegalArgumentException;

    /**
     * Vráti skupinu všetkých hrán incidentných so zadaným vrcholom.
     * @param vertex Vrchol ktorého hrany žiadame.
     * @throws IllegalArgumentException ak taký vrchol v grafe neexistuje
     */
    public Collection<Edge> adjEdges(Vertex vertex)
    throws IllegalArgumentException;

    /**
     * Vráti skupinu všetkých hrán incidentných s vrcholom zadaným
     * pomocou id.
     * @param id id zadaného vrcholu
     * @throws IllegalArgumentException ak vrchol so zadaným id neexistuje
     */
    public Collection<Edge> adjEdges(int id) throws IllegalArgumentException;

    /**
     * Do grafu pridá vrchol so zadanými súradnicami. Súradnice majú byť
     * medzi 0 a 1. Hondoty mimo tohto intervalu budú upravené na 0 alebo na 1.
     * @param x súradnica x vrchola (číslo od 0 po 1)
     * @param y súradnica y vrchola (číslo od 0 po 1)
     * @return pridaný vrchol
     */
    public Vertex addVertex(double x, double y);

    /**
     * Vymaže zadaný vrchol. Ako dôsledok prečísluje vrcholy s vyššími číslami.
     * @param vertex vymazávaný vrchol
     * @throws IllegalArgumentException ak taký vrchol v grafe neexistuje
     */
    public void removeVertex(Vertex vertex)
    throws IllegalArgumentException;


    /**
     * Pridá hranu medzi vrcholmi vertex1 a vertex2.
     * Farba a hodnota hrany budú prednastavené hodnoty.
     * @param vertex1 počiatočný vrchol hrany
     * @param vertex2 koncový vrchol hrany
     * @return Vytvorená hrana z vertex1 do vertex2.
     * @throws IllegalArgumentException ak taká hrana už existuje,
     * alebo ak vertex1=vertex2 alebo ak vertex1 alebo vertex2 nie sú
     * v grafe.
     */
    public Edge addEdge(Vertex vertex1, Vertex vertex2)
    throws IllegalArgumentException;

    /**
     * Pridá hranu medzi vrcholmi s id číslami id1 a id2.
     * @param id1 id počiatočného vrchola
     * @param id2 id koncového vrchola
     * @return Vytvorená hrana z id1 do id2.
     * @throws IllegalArgumentException ak taká hrana už existuje
     * alebo ak id1 = id2 alebo ak id1 alebo id2 nie sú platné id vrcholov
     */
    public Edge addEdge(int id1, int id2) throws IllegalArgumentException;

    /**
     * Vymaže hranu z grafu.
     * @param edge Hrana, ktorú chceme zmazať.
     * @throws IllegalArgumentException ak taká hrana neexistuje
     */
    public void removeEdge(Edge edge) throws IllegalArgumentException;

    /**
     * Vymaže hranu medzi zadanými vrcholmi z grafu.
     * @param vertex1 koncový vrchol hrany
     * @param vertex2 koncový vrchol hrany
     * @throws IllegalArgumentException ak taká hrana neexistuje
     */
    public void removeEdge(Vertex vertex1, Vertex vertex2)
    throws IllegalArgumentException;

    /**
     * Vymaže hranu medzi vrcholmi s id číslami id1 a id2 z grafu,
     * @param id1 id vrchola
     * @param id2 id vrchola
     * @throws IllegalArgumentException ak taká hrana neexistuje
     */
    public void removeEdge(int id1, int id2) throws IllegalArgumentException;

    /**
     * Nájde hranu medzi zadanými vrcholmi.
     * @param vertex1 vrchol
     * @param vertex2 vrchol
     * @return nájdená hrana, alebo null ak neexistuje
     * @throws IllegalArgumentException ak taká vrcholy neexistujú v grafe
     */
    public Edge findEdge(Vertex vertex1, Vertex vertex2)
    throws IllegalArgumentException;

    /**
     * Nájde hranu medzi vrcholmi s id číslom id1 a id2.
     * @param id1 id vrchola
     * @param id2 id vrchola
     * @return nájdená hrana, alebo null ak neexistuje
     * @throws IllegalArgumentException ak také vrcholy neexistujú
     */
    public Edge findEdge(int id1, int id2) throws IllegalArgumentException;

    /**
     * Vymaže všetky vrcholy a hrany z grafu.
     */
    public void clear();

    /**
     * Vypíše graf do výstupného súboru.
     * @param out Printstream, kam sa vypisuje graf.
     * @param full Ak je nastavené na true, vypíšu sa aj farby
     *     a veľkosti vrcholov.
     */
    public void print(PrintStream out, boolean full);


    /**
     * Načíta graf zo Scannera s.
     * Všetky predtým existujúce vrcholy a hrany budú zmazané.
     *
     * @param scanner Scanner zo vstupného súboru
     * @throws InputMismatchException v prípade, že formát vstupu nesedí
     */
    public void read(Scanner scanner)
    throws InputMismatchException;

    /**
     * Zruší výber hrany.
     */
    public void deselectEdge();

    /**
     * Zruší výber vrcholu.
     */
    public void deselectVertex();


    /**
    * Označí vrchol.
    * @param vertex Vrchol, ktorý chceme označiť
    * @throws IllegalArgumentException ak taký vrchol v grafe neexistuje.
    */
    public void selectVertex(Vertex vertex)
    throws IllegalArgumentException;

    /**
     * Označí hranu.
     * @param edge Hrana, ktorú chceme označiť.
     * @throws IllegalArgumentException ak taká hrana v grafe neexistuje
     */
    public void selectEdge(Edge edge)
    throws IllegalArgumentException;

    /**
     * Vráti označený vrchol alebo null, ak žiaden vrchol nie je označený.
     * @return označený vrchol alebo null
     */
    public Vertex getSelectedVertex();

    /**
     * Vráti označenú hranu alebo null, ak žiadna hrana nie je označená.
     * @return označená hrana alebo null
     */
    public Edge getSelectedEdge();

}
