package graphgui;

import java.util.HashMap;
import javafx.scene.paint.Color;
import javafx.scene.shape.Circle;
import javafx.scene.shape.Rectangle;
import javafx.scene.text.Text;
import javafx.scene.shape.Line;
import javafx.scene.shape.Shape;
import javafx.scene.layout.Pane;
import javafx.scene.text.Font;
import javafx.scene.text.TextAlignment;
import javafx.scene.text.TextBoundsType;
import javafx.geometry.VPos;
import javafx.geometry.Bounds;

/**
 * Trieda pre vykresľovanie vrcholov a hrán grafu. Pre každú
 * hranu a vrchol si pamätá ich Shape (Circle, Line). Túto triedu
 * priamo nepoužívajte vo vašej časti programu, ku grafom pristupujte
 * len pomocou rozhraní Graph, Vertex, Edge.
 */
public class GraphDrawing implements GraphImplementation.GraphObserver {

    private final Pane pane;

    private static final double VERTEX_RADIUS = 15;
    private static final double BORDER = VERTEX_RADIUS + 3;
    private static final int LINE_WIDTH = 3;
    private static final int SELECTED_LINE_WIDTH = 6;
    private static final double TEXT_FRACTION = 0.8;

    private final HashMap<Vertex, Circle> vertexShapes;
    private final HashMap<Vertex, Text> vertexTexts;
    private final HashMap<Edge, Line> edgeShapes;


    /**
     * Konštruktor.
     */
    public GraphDrawing(Pane pane) {
        this.pane = pane;
        vertexShapes = new HashMap<>();
        vertexTexts = new HashMap<>();
        edgeShapes = new HashMap<>();
        State.getGraph().addObserver(this);
    }

    @Override
    public void vertexAdded(Vertex v) {
        vertexShapes.put(v, new Circle());
        vertexTexts.put(v, new Text());
        updateVertexShape(v);
        pane.getChildren().add(getVertexCircle(v));
        pane.getChildren().add(getVertexText(v));
    }

    @Override
    public void vertexRemoved(Vertex v) {
        pane.getChildren().remove(getVertexCircle(v));
        pane.getChildren().remove(getVertexText(v));
        vertexShapes.remove(v);
        vertexTexts.remove(v);

    }

    @Override
    public void edgeAdded(Edge e) {
        edgeShapes.put(e, new Line());
        updateEdgeShape(e);
        Shape s = getEdgeShape(e);
        pane.getChildren().add(s);
        s.toBack();
    }

    @Override
    public void edgeRemoved(Edge e) {
        pane.getChildren().remove(getEdgeShape(e));
        edgeShapes.remove(e);
    }

    @Override
    public void vertexChanged(Vertex v) {
        updateVertexShape(v);
    }

    @Override
    public void edgeChanged(Edge e) {
        updateEdgeShape(e);
    }

    private void updateVertexShape(Vertex v) {
        GraphImplementation graph = State.getGraph();
        Circle circle = vertexShapes.get(v);
        double x = getVertexX(v);
        double y = getVertexY(v);

        circle.setCenterX(x);
        circle.setCenterY(y);
        circle.setRadius(VERTEX_RADIUS);
        circle.setFill(Color.web(v.getColorName()));
        circle.setStroke(Color.web("black"));

        if (v == graph.getSelectedVertex()) {
            circle.setStrokeWidth(SELECTED_LINE_WIDTH);
        } else {
            circle.setStrokeWidth(LINE_WIDTH);
        }


        Text text = vertexTexts.get(v);
        String id = String.valueOf(v.getId());
        Rectangle r = new Rectangle
        (circle.getCenterX() - circle.getRadius() * TEXT_FRACTION,
         circle.getCenterY() - circle.getRadius() * TEXT_FRACTION,
         circle.getRadius() * TEXT_FRACTION * 2,
         circle.getRadius() * TEXT_FRACTION * 2);

        updateBoundedText(text, id, r, Color.BLACK);

        for (Edge e : v.adjEdges()) {
            updateEdgeShape(e);
        }
    }

    private void updateEdgeShape(Edge e) {
        GraphImplementation graph = State.getGraph();
        Line l = edgeShapes.get(e);
        l.setStartX(getVertexX(e.getFirst()));
        l.setStartY(getVertexY(e.getFirst()));
        l.setEndX(getVertexX(e.getSecond()));
        l.setEndY(getVertexY(e.getSecond()));
        l.setStroke(Color.web(e.getColorName()));

        if (e == graph.getSelectedEdge()) {
            //l.getStrokeDashArray().clear();
            //l.getStrokeDashArray().addAll(2d, 10d);
            l.setStrokeWidth(GraphDrawing.SELECTED_LINE_WIDTH);
        } else {
            //l.getStrokeDashArray().clear();
            l.setStrokeWidth(GraphDrawing.LINE_WIDTH);
        }
    }

    public void updateAll() {
        Graph graph = State.getGraph();
        for (Vertex vertex : graph.getVertices()) {
            updateVertexShape(vertex);
        }
        for (Edge edge : graph.getEdges()) {
            updateEdgeShape(edge);
        }
    }

    public Circle getVertexCircle(Vertex v) {
        return vertexShapes.get(v);
    }

    public Text getVertexText(Vertex v) {
        return vertexTexts.get(v);
    }

    public Shape getEdgeShape(Edge e) {
        return edgeShapes.get(e);
    }

    @Override
    public void edgeDeselected(Edge e) {
        updateEdgeShape(e);
    }

    @Override
    public void vertexDeselected(Vertex v) {
        updateVertexShape(v);
    }

    @Override
    public void vertexSelected(Vertex v) {
        updateVertexShape(v);
    }

    @Override
    public void  edgeSelected(Edge e) {
        updateEdgeShape(e);
    }

    /**
     * Metoda ktora nam vrati Text ktory bude napchany v obldzniku ktory mu zadame.
     * @param bounds Obldznik ktory ohranicuje rozmery Text-u
     * @param c Farba textu
     * @param s Samotny text
     *
     * @return Text s co najvacsim pismom vopchany do zadaneho obdlznika
     */
    private static void updateBoundedText(Text text,
                                          String s, Rectangle bounds, Color c) {
        text.setText(s);
        text.setTextAlignment(TextAlignment.CENTER);
        text.setTextOrigin(VPos.CENTER);
        text.setFill(c);
        // nastavime na velke pismo a potom zoskalujeme
        text.setFont(new Font("Verdana", 100));
        text.setBoundsType(TextBoundsType.LOGICAL_VERTICAL_CENTER);
        Bounds textBounds = text.getBoundsInLocal();
        double scale = Math.min(bounds.getWidth() / textBounds.getWidth(),
                                bounds.getHeight() / (text.getLayoutBounds().getHeight()));
        //text.setScaleX(bounds.getWidth() / textBounds.getWidth());
        //text.setScaleY(bounds.getHeight() / textBounds.getHeight());

        text.setScaleX(scale);
        text.setScaleY(scale);

        // pocas skalovania sa menia "layout bounds"
        text.setX(
            bounds.getX() + bounds.getWidth() / 2 - text.getLayoutBounds().getWidth() / 2);
        text.setY(bounds.getY() + bounds.getHeight() / 2);
        text.setPickOnBounds(true);
        text.setMouseTransparent(true);


    }

    public double backTransformX(double x) {
        double max = pane.getWidth() - 2 * BORDER;
        return (x - BORDER) / max;
    }

    public double backTransformY(double y) {
        double max = pane.getHeight() - 2 * BORDER;
        return (y - BORDER) / max;
    }

    public double transformX(double x) {
        double max = pane.getWidth() - 2 * BORDER;
        return x * max + BORDER;
    }

    public double transformY(double y) {
        double max = pane.getHeight() - 2 * BORDER;
        return y * max + BORDER;
    }

    public double getVertexX(Vertex vertex) {
        return transformX(vertex.getX());
    }

    public double getVertexY(Vertex vertex) {
        return transformY(vertex.getY());
    }
}
