package graphgui;

import java.io.PrintStream;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Collection;
import java.util.Scanner;
import java.util.InputMismatchException;

/**
 * Trieda GraphImplementation implementuje rozhranie Graph na prácu s grafmi.
 * V svojej časti programu pristupujte ku grafu len cez
 * rozhrania Graph, Vertex a Edge.
 */
public class GraphImplementation implements Graph {

    private final ArrayList<VertexImplementation> vertices;
    private final ArrayList<EdgeImplementation> edges;

    private Edge selectedEdge = null;  // nemusi byt primary
    private Vertex selectedVertex = null;

    /**
     * Pomocné rozhranie, určené pre triedy, ktoré chcú byť upovedomené o
     * zmenách v grafe, aby ich mohli zobraziť.
     */
    public interface GraphObserver {
        public void vertexAdded(Vertex vertex);
        public void edgeAdded(Edge edge);
        public void vertexRemoved(Vertex vertex);
        public void edgeRemoved(Edge edge);
        public void vertexChanged(Vertex vertex);
        public void edgeChanged(Edge edge);
        public void vertexDeselected(Vertex vertex);
        public void vertexSelected(Vertex vertex);
        public void edgeDeselected(Edge edge);
        public void edgeSelected(Edge edge);
    }

    private final ArrayList<GraphObserver> observers;


    /** Konštruktor vytvorí prázdny graf. */
    public GraphImplementation() {
        vertices = new ArrayList<>();
        edges = new ArrayList<>();
        observers = new ArrayList<>();
    }

    public void addObserver(GraphObserver o) {
        observers.add(o);
    }
    public void removeObserver(GraphObserver o) {
        observers.remove(o);
    }


    @Override
    public int getNumberOfVertices() {
        return vertices.size();
    }

    @Override
    public Vertex getVertex(int index) throws IllegalArgumentException {
        if (index < 0 || index >= vertices.size()) {
            throw new IllegalArgumentException("vertex id " + index + " does not exists");
        }
        return vertices.get(index);
    }

    @Override
    public VertexImplementation addVertex(double x, double y) {
        VertexImplementation v = new VertexImplementation(this, vertices.size(), x, y);
        vertices.add(v);
        for (GraphObserver o : observers) {
            o.vertexAdded(v);
        }
        return v;
    }

    @Override
    public void removeVertex(Vertex vertex) {
        checkVertex(vertex);
        for (Edge e : vertex.adjEdges()) {
            removeEdge(e);
        }
        if (selectedVertex == vertex) {
            deselectVertex();
        }
        boolean wasDeleted = vertices.remove((VertexImplementation)vertex);
        if (!wasDeleted) {
            throw new IllegalArgumentException("no such vertex");
        }
        internalRecomputeVertexIds();
        for (GraphObserver o : observers) {
            o.vertexRemoved(vertex);
        }
    }

    @Override
    public Collection<Vertex> getVertices() {
        return Collections.unmodifiableList(new ArrayList<Vertex>(vertices));
    }

    @Override
    public Collection<Vertex> adjVertices(Vertex vertex) {
        checkVertex(vertex);
        return vertex.adjVertices();
    }

    @Override
    public Collection<Edge> adjEdges(Vertex vertex) {
        checkVertex(vertex);
        return vertex.adjEdges();
    }

    @Override
    public Collection<Edge> adjEdges(int n) throws IllegalArgumentException {
        return getVertex(n).adjEdges();
    }

    @Override
    public Collection<Integer> adjVertexIds(int n) throws IllegalArgumentException {
        return getVertex(n).adjVertexIds();
    }

    private void checkVertex(Vertex v) throws IllegalArgumentException {
        int id = v.getId();
        if (id < 0 || id >= getNumberOfVertices() ||  vertices.get(id) != v) {
            throw new IllegalArgumentException("vertex does not exist in the graph");
        }
    }

    @Override
    public EdgeImplementation addEdge(Vertex v1, Vertex v2) throws IllegalArgumentException {
        if (findEdge(v1, v2) != null) {
            throw new IllegalArgumentException("edge already exists");
        }
        if (v1 == v2) {
            throw new IllegalArgumentException("origin and destination are the same");
        }
        VertexImplementation vi1 = (VertexImplementation)v1;
        VertexImplementation vi2 = (VertexImplementation)v2;
        EdgeImplementation edge = new EdgeImplementation(this, vi1, vi2);
        vi1.internalAddEdge(edge);
        vi2.internalAddEdge(edge);
        edges.add(edge);
        for (GraphObserver o : observers) {
            o.edgeAdded(edge);
        }
        return edge;
    }

    @Override
    public Edge addEdge(int n1, int n2)
    throws IllegalArgumentException
    {
        return addEdge(getVertex(n1), getVertex(n2));
    }


    @Override
    public void removeEdge(Edge e) throws IllegalArgumentException {
        if (selectedEdge != null && selectedEdge == e) {
            deselectEdge();
        }

        EdgeImplementation ei = (EdgeImplementation)e;

        boolean wasRemoved = edges.remove(ei);
        if (!wasRemoved) {
            throw new IllegalArgumentException("no such edge");
        }

        VertexImplementation vi1 = ei.getFirst();
        VertexImplementation vi2 = ei.getSecond();

        wasRemoved = vi1.internalRemoveEdge(ei);
        wasRemoved = wasRemoved && vi2.internalRemoveEdge(ei);
        if (!wasRemoved) {
            throw new IllegalArgumentException("graph inconsistent");
        }

        for (GraphObserver o : observers) {
            o.edgeRemoved(e);
        }
    }

    @Override
    public void removeEdge(Vertex n1, Vertex n2) throws IllegalArgumentException {
        Edge e = findEdge(n1, n2);
        if (e == null) {
            throw new IllegalArgumentException("no such edge");
        }
        removeEdge(e);
    }

    @Override
    public void removeEdge(int n1, int n2) throws IllegalArgumentException {
        removeEdge(getVertex(n1), getVertex(n2));
    }

    @Override
    public Edge findEdge(Vertex v1, Vertex v2) {
        checkVertex(v1);
        checkVertex(v2);
        return v1.findEdge(v2);
    }

    @Override
    public Edge findEdge(int n1, int n2) throws IllegalArgumentException {
        return findEdge(getVertex(n1), getVertex(n2));
    }

    @Override
    public Collection<Edge> getEdges() {
        return Collections.unmodifiableList(new ArrayList<Edge>(edges));
    }

    @Override
    public int getNumberOfEdges() {
        return edges.size();
    }

    void internalRecomputeVertexIds() {
        int id = 0;
        for (VertexImplementation n : vertices) {
            n.internalSetId(id++);
        }
    }

    @Override
    public void print(PrintStream out, boolean full) {
        out.print("clear\n");
        for (Vertex n : getVertices()) {
            out.format("add vertex %d x %g y %g", n.getId(),
                       n.getX(), n.getY());
            if (full) {
                out.format(" color %s value %d",
                           n.getColorName(), n.getValue());
            }
            out.println();
        }
        for (Edge e : getEdges()) {
            out.format("add edge %d %d", e.getFirstId(),
                       e.getSecondId());
            if (full) {
                out.format(" color %s value %d", e.getColorName(),
                           e.getValue());
            }
            out.println();
        }
        if (getSelectedVertex() != null) {
            out.format("select vertex %d\n", getSelectedVertex().getId());
        }
        if (getSelectedEdge() != null) {
            Edge e = getSelectedEdge();
            out.format("select edge %d %d\n", e.getSecondId(),
                       e.getFirstId());
        }
    }

    @Override
    public void clear() {
        for (Vertex n : getVertices()) {
            removeVertex(n);
        }
    }

    @Override
    public void read(Scanner s) throws InputMismatchException {
        clear();  // remove all existing vertices
        int lineNum = 0;
        while (s.hasNextLine()) {
            String line = s.nextLine();
            lineNum++;

            try {
                runCommand(line);
            } catch (Exception e) {
                String trimmed = line.replaceAll("\n", "");
                String message
                    = String.format("bad line %d '%s' (%s)",
                                    lineNum, trimmed, e.getMessage());
                throw new InputMismatchException(message);
            }
        }
    }

    /**
     * Pomocná metóda, ktorou Vertex oznamuje svoje zmeny.
     */
    public void graphVertexChanged(Vertex v) {
        for (GraphObserver o : observers) {
            o.vertexChanged(v);
        }
    }

    /**
    * Pomocná metóda, ktorou Edge oznamuje svoje zmeny.
    */
    public void graphEdgeChanged(Edge e) {
        for (GraphObserver o : observers) {
            o.edgeChanged(e);
        }
    }


    @Override
    public void deselectEdge() {
        if (selectedEdge != null) {
            Edge old = selectedEdge;
            selectedEdge = null;
            for (GraphObserver o : observers) {
                o.edgeDeselected(old);
            }
        }
    }


    @Override
    public void selectVertex(Vertex v) {
        if (v == null) {
            deselectVertex();
        }
        if (v != selectedVertex) {
            deselectVertex();
            selectedVertex = v;
            for (GraphObserver o : observers) {
                o.vertexSelected(selectedVertex);
            }
        }
    }


    @Override
    public void selectEdge(Edge e) {
        if (e == null) {
            deselectEdge();
        }
        if (e != selectedEdge) {
            deselectEdge();
            selectedEdge = e;
            for (GraphObserver o : observers) {
                o.edgeSelected(selectedEdge);
            }
        }
    }

    @Override
    public void deselectVertex() {
        if (selectedVertex != null) {
            Vertex old = selectedVertex;
            selectedVertex = null;
            for (GraphObserver o : observers) {
                o.vertexDeselected(old);
            }
        }
    }

    @Override
    public Vertex getSelectedVertex() {
        return selectedVertex;
    }

    @Override
    public Edge getSelectedEdge() {
        return selectedEdge;
    }

    static private class VertexProperties {
        public Integer id = null;
        public String color = null;
        public Integer value = null;
        public Double x = null;
        public Double y = null;

        VertexProperties(Scanner s, Graph graph) {
            if (s.hasNextInt()) {
                id = s.nextInt();
            } else {
                if (graph != null && graph.getSelectedVertex() != null) {
                    id = graph.getSelectedVertex().getId();
                }
            }
            while (s.hasNext()) {
                String property = s.next();
                try {
                    switch (property) {
                    case "color":
                        color = s.next();
                        break;
                    case "x":
                        x = s.nextDouble();
                        break;
                    case "y":
                        y = s.nextDouble();
                        break;
                    case "value":
                        value = s.nextInt();
                        break;
                    default:
                        throw new InputMismatchException();
                    }
                } catch (Exception e) {
                    throw new InputMismatchException("error while parsing vertex property " + property);
                }
            }
        }

        public void updateValue(Vertex vertex) {
            if (value != null) {
                vertex.setValue(value);
            }
        }

        public void updateColor(Vertex vertex) {
            if (color != null) {
                vertex.setColorName(color);
            }
        }

        public void updateCoord(Vertex vertex) {
            if (x != null) {
                vertex.setX(x);
            }
            if (y != null) {
                vertex.setY(y);
            }
        }
    }

    static private class EdgeProperties {
        public Vertex from = null;
        public Vertex to = null;
        public String color = null;
        public Integer value = null;

        EdgeProperties(Scanner s, Graph graph) {
            if (s.hasNextInt()) {
                Integer id1 = s.nextInt();
                Integer id2;
                if (s.hasNextInt()) {
                    id2 = s.nextInt();
                } else {
                    id2 = id1;
                    Vertex selected = graph.getSelectedVertex();
                    if (selected == null) {
                        throw new InputMismatchException("no vertex selected and only one edge endpoint given");
                    }
                    id1 = selected.getId();
                }
                from = graph.getVertex(id1);
                to = graph.getVertex(id2);
            } else {
                // ziadne cislo
                Edge selected = graph.getSelectedEdge();
                if (selected == null) {
                    throw new InputMismatchException("no edge selected and no endpoint given");
                }
                from = selected.getFirst();
                to = selected.getSecond();
            }

            while (s.hasNext()) {
                String property = s.next();
                try {
                    switch (property) {
                    case "color":
                        color = s.next();
                        break;
                    case "value":
                        value = s.nextInt();
                        break;
                    default:
                        throw new InputMismatchException();
                    }
                } catch (Exception e) {
                    throw new InputMismatchException("error while parsing vertex property " + property);
                }
            }
        }

        public void updateValue(Edge edge) {
            if (value != null) {
                edge.setValue(value);
            }
        }

        public void updateColor(Edge edge) {
            if (color != null) {
                edge.setColorName(color);
            }
        }
    }

    public String runCommand(String command) {
        if (command.length() == 0 || command.charAt(0) == '#') {
            return "";
        }
        Scanner s = new Scanner(command);
        if (!s.hasNext()) { // prazdny prikaz
            return "";
        }

        String commandName = s.next();
        if (commandName.equals("add")
                || commandName.equals("remove")
                || commandName.equals("select")
                || commandName.equals("update")
                || (commandName.equals("deselect") && s.hasNext())) {
            if (!s.hasNext()) {
                throw new InputMismatchException("missing part of command name");
            }
            commandName += " " + s.next();
        }
        if (commandName.equals("add vertex")) {
            VertexProperties prop = new VertexProperties(s, null);
            if (prop.x == null) {
                prop.x = Math.random();
            }
            if (prop.y == null) {
                prop.y = Math.random();
            }

            Vertex vertex = this.addVertex(prop.x, prop.y);
            prop.updateValue(vertex);
            prop.updateColor(vertex);
            if (prop.id != null && vertex.getId() != prop.id) {
                throw new InputMismatchException(String.format("new vertex id %d not %d", prop.id, vertex.getId()));
            }
            return String.format("vertex %d added (x %f y %f value %d color %s)", vertex.getId(),
                                 vertex.getX(), vertex.getY(), vertex.getValue(), vertex.getColorName());
        }

        if (commandName.equals("update vertex")) {
            VertexProperties prop = new VertexProperties(s, this);
            if (prop.id == null) {
                throw new InputMismatchException("vertex id not given and no vertex selected");
            }
            Vertex vertex = this.getVertex(prop.id);
            prop.updateValue(vertex);
            prop.updateColor(vertex);
            prop.updateCoord(vertex);
            return String.format("vertex %d updated (x %f y %f value %d color %s)", vertex.getId(),
                                 vertex.getX(), vertex.getY(), vertex.getValue(), vertex.getColorName());
        }

        if (commandName.equals("remove vertex")) {
            VertexProperties prop = new VertexProperties(s, this);
            if (prop.id == null) {
                throw new InputMismatchException("vertex id not given and no vertex selected");
            }
            removeVertex(getVertex(prop.id));
            return "vertex " + prop.id + " removed";
        }

        if (commandName.equals("select vertex")) {
            VertexProperties prop = new VertexProperties(s, this);
            if (prop.id == null) {
                throw new InputMismatchException("vertex id not given");
            }
            selectVertex(getVertex(prop.id));
            return "vertex" + prop.id + " selected";
        }

        if (commandName.equals("deselect vertex")) {
            deselectVertex();
            return "vertex deselected";
        }

        if (commandName.equals("add edge")) {
            EdgeProperties prop = new EdgeProperties(s, this);
            Edge edge = this.addEdge(prop.from, prop.to);
            prop.updateValue(edge);
            prop.updateColor(edge);
            return String.format("edge %d %d added (value %d color %s)",
                                 edge.getFirstId(), edge.getSecondId(),
                                 edge.getValue(), edge.getColorName());
        }

        if (commandName.equals("update edge")) {
            EdgeProperties prop = new EdgeProperties(s, this);
            Edge edge = findEdge(prop.from, prop.to);
            if (edge == null) {
                throw new InputMismatchException("edge does not exist");
            }
            prop.updateValue(edge);
            prop.updateColor(edge);
            return String.format("edge %d %d updated (value %d color %s)",
                                 edge.getFirstId(), edge.getSecondId(),
                                 edge.getValue(), edge.getColorName());
        }

        if (commandName.equals("remove edge")) {
            EdgeProperties prop = new EdgeProperties(s, this);
            removeEdge(prop.from, prop.to);
            return String.format("edge %d %d removed", prop.from.getId(), prop.to.getId());
        }

        if (commandName.equals("select edge")) {
            EdgeProperties prop = new EdgeProperties(s, this);
            Edge edge = findEdge(prop.from, prop.to);
            if (edge == null) {
                throw new InputMismatchException("edge does not exist");
            }
            selectEdge(edge);
            return  String.format("edge %d %d selected", edge.getFirstId(), edge.getSecondId());
        }

        if (commandName.equals("deselect edge")) {
            deselectEdge();
            return "edge deselected";
        }

        if (commandName.equals("deselect")) {
            deselectVertex();
            deselectEdge();
            return "vertex and edge deselected";
        }

        if (commandName.equals("clear")) {
            clear();
            return "all vertices and edges removed";
        }

        if (commandName.equals("save")) {
            String filename = s.nextLine().trim();
            PrintStream out = null;
            try {
                out = new PrintStream(filename);
            } catch (Exception e) {
                throw new InputMismatchException("cannot open file '" + filename + "'");
            }
            boolean longFormat = true;
            if (s.hasNext() && s.next().equals("short")) {
                longFormat = false;
            }
            print(out, longFormat);
            out.close();
            return "graph saved to file " + filename;
        }

        if (commandName.equals("open")) {
            String filename = s.nextLine().trim();
            Scanner in = null;
            try {
                in = new Scanner(new File(filename));
            } catch (Exception e) {
                throw new InputMismatchException("cannot open file " + filename);
            }
            read(in);
            in.close();
            return "graph read from file " + filename;
        }

        throw new InputMismatchException("bad command " + command);
    }

}
