package graphgui;

import javafx.geometry.Insets;
import javafx.scene.Cursor;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.Background;
import javafx.scene.layout.BackgroundFill;
import javafx.scene.layout.CornerRadii;
import javafx.scene.layout.Pane;
import javafx.scene.paint.Color;
import javafx.scene.shape.Circle;
import javafx.scene.shape.Line;
import javafx.scene.shape.Shape;

/** Trieda pre grafický panel, na ktorom umiestňujeme nakreslenie grafu. */
public class GraphPane extends Pane implements GraphImplementation.GraphObserver {

    private Controller controller;
    private GraphDrawing drawing;
    private final Line edgeLine;

    /**
     * Inicializuje grafový panel.
     */
    public GraphPane() {
        this.edgeLine = new Line();
        this.getChildren().add(this.edgeLine);
        this.setMinSize(400, 400);
        this.setBackground(new Background(new BackgroundFill(
                                              Color.LIGHTGRAY, CornerRadii.EMPTY, Insets.EMPTY
                                          )));
        this.toBack();
    }

    /**
     * Inicializácia pri naloadovaní FXML.
     * @param controller Controller okna aplikácie
     */
    public void init(Controller controller) {
        this.controller = controller;
        this.drawing =  new GraphDrawing(this);
        State.getGraph().addObserver(this);
        updateLine(null);

        this.setOnMouseMoved((MouseEvent e) -> {
            updateLine(e);
        });

        this.setOnMouseClicked((MouseEvent event) -> {
            if (State.getMode() == State.GraphMode.ADD
            && !this.isAddingEdge()) {
                String command = String.format("add vertex x %f y %f",
                drawing.backTransformX(event.getX()),
                drawing.backTransformX(event.getY()));
                controller.runCommand(command, false);
            } else if (State.getMode() == State.GraphMode.SELECT) {
                controller.runCommand("deselect", false);
            }
        });

        // zmena velkosti pane - poposuvaj vrcholy
        this.widthProperty().addListener((obs, oldVal, newVal) -> {
            drawing.updateAll();
        });
        this.heightProperty().addListener((obs, oldVal, newVal) -> {
            drawing.updateAll();
        });
    }

    private boolean isAddingEdge() {
        return State.getMode() == State.GraphMode.ADD && State.getGraph().getSelectedVertex() != null;
    }

    /**
     * Prekresli ciaru aby viedla od suradnic vrchola po poziciu mysky.
     * @param event udalost obsahujuca poziciu mysky
     */
    public void updateLine(MouseEvent event) {
        if (isAddingEdge() && event != null) {
            Vertex v = State.getGraph().getSelectedVertex();
            this.edgeLine.setStartX(drawing.getVertexX(v));
            this.edgeLine.setStartY(drawing.getVertexY(v));
            this.edgeLine.setEndX(event.getX());
            this.edgeLine.setEndY(event.getY());
        } else {
            // "Zmaže" Shape novej editovacej hrany posunutím do rohu
            // a nastavením dĺžky na 0.
            this.edgeLine.setStartX(0);
            this.edgeLine.setStartY(0);
            this.edgeLine.setEndX(0);
            this.edgeLine.setEndY(0);
        }
    }

    private void toggleVertexSelection(Vertex vertex) {
        if (vertex == State.getGraph().getSelectedVertex()) {
            controller.runCommand("deselect vertex", false);
        } else {
            controller.runCommand("select vertex " + vertex.getId(), false);
        }
    }

    private void toggleEdgeSelection(Edge edge) {
        if (edge == State.getGraph().getSelectedEdge()) {
            controller.runCommand("deselect edge", false);
        } else {
            String command = String.format("select edge %d %d",
                                           edge.getFirstId(),
                                           edge.getSecondId());
            controller.runCommand(command, false);
        }
    }

    /**
     * Pridá vrchol do panelu.
     * @param vertex Pridaný vrchol
     * @throws IllegalArgumentException ak vrchol nebol korektný
     */
    @Override
    public void vertexAdded(Vertex vertex) throws IllegalArgumentException {
        Circle circle = drawing.getVertexCircle(vertex);
        circle.setCursor(Cursor.HAND);

        circle.setOnMouseClicked((MouseEvent event) -> {
            Graph graph = State.getGraph();
            event.consume();
            switch (State.getMode()) {
            case DELETE:
                controller.runCommand("remove vertex " + vertex.getId(), false);
                break;
            case ADD:
                Vertex selected = graph.getSelectedVertex();
                if (selected != null && selected != vertex) {
                    if (graph.findEdge(selected, vertex) == null) {
                        String command = String.format("add edge %d %d", selected.getId(), vertex.getId());
                        controller.runCommand(command, false);
                        controller.runCommand("deselect vertex", false);
                    }
                } else {
                    toggleVertexSelection(vertex);
                }
                break;
            case SELECT:
                toggleVertexSelection(vertex);
                break;
            default:
                break;
            }
        });

        circle.setOnMouseMoved((MouseEvent event) -> {
            updateLine(event);
        });
    }

    /**
     * Pridá hranu do panelu.
     * @param edge Pridaná hrana
     */
    @Override
    public void edgeAdded(Edge edge) throws IllegalArgumentException {
        Shape s = drawing.getEdgeShape(edge);
        s.setCursor(Cursor.HAND);

        s.setOnMouseClicked((MouseEvent event) -> {
            event.consume();
            State.GraphMode mode = State.getMode();
            if (mode == State.GraphMode.SELECT) {
                toggleEdgeSelection(edge);
            } else if (mode == State.GraphMode.DELETE) {
                String command = String.format("remove edge %d %d",
                edge.getFirstId(),
                edge.getSecondId());
                controller.runCommand(command, false);
            }
        });
    }

    /**
     * Zmaže vrchol z panelu.
     * @param vertex Zmazaný vrchol
     */
    @Override
    public void vertexRemoved(Vertex vertex) {
    }

    @Override
    public void edgeRemoved(Edge edge) {
    }

    @Override
    public void edgeSelected(Edge edge) {
    }

    @Override
    public void edgeDeselected(Edge edge) {
    }

    @Override
    public void vertexSelected(Vertex vertex) {
    }

    @Override
    public void vertexDeselected(Vertex vertex) {
        this.updateLine(null);
    }

    @Override
    public void edgeChanged(Edge edge) {
    }

    @Override
    public void vertexChanged(Vertex vertex) {
    }

}
