package graphgui;

import java.util.Collection;

/**
 * Rozhranie pre vrchol grafu.
 * Každý vrchol má svoj identifikátor (id),
 * čo je číslo od 0 po N-1, kde N je počet vrcholov grafu.
 * Tento identifikátor sa môže po vymazaní vrcholu z grafu meniť.
 * Vo vrchole je tiež uložená celočíslená hodnota (value).
 * Napokon, vzhľad vrcholu je určený jeho farbou (color),
 * a polohou (position).
 * Farba je daná ako reťazec obsahujúci buď meno farby podľa
 * HTML a CSS špecifikácie, napr. "white", alebo RGB hodnotu.
 */
public interface Vertex {

    /** Vráti id vrcholu (číslo od 0 po N-1, kde N je počet vrcholov grafu). */
    int getId();

    /** Vráti skupinu vrcholov susediacich s vrcholom this. */
    public Collection<Vertex> adjVertices();

    /** Vráti skupinu identifikátorov vrcholov susediacich s vrcholom this. */
    public Collection<Integer> adjVertexIds();

    /** Vráti skupinu hrán incidentných s vrcholom this.
     */
    public Collection<Edge> adjEdges();

    /** Nájde hranu z vrcholu this do zadaného vrcholu vertex.
     * @param vertex vrchol, do ktorého hranu hľadáme
     * @return nájdená hrana, alebo null ak neexistuje
     */
    Edge findEdge(Vertex vertex);

    /** Vráti celočíselnú hodnotu uloženú vo vrchole. */
    int getValue();

    /**
     * Nastaví hodnotu vrcholu na zadané číslo.
     * @param value nová hodnota vrcholu.
     */
    void setValue(int value);

    /** Vráti meno farby vrcholu ako String. */
    public String getColorName();

    /**
     * Nastaví novú farbu vrcholu.
     * @param color meno požadovanej novej farby pre vrchol.
     * @throws IllegalArgumentException ak meno nereprezentuje platnú farbu
     */
    public void setColorName(String color)
    throws IllegalArgumentException;

    /**  Nastaví novú farbu hrany z rgb hodnôt. Tieto hodnoty majú byť
     *  od 0 po 255. Ak dostane čísla mimo rozsahu, hodí výnimku.
     * @param red intenzita červenej (číslo od 0 po 255)
     * @param green intenzita zelenej (číslo od 0 po 255)
     * @param blue intenzita modrej (číslo od 0 po 255)
     * @throws IllegalArgumentException ak hodnoty mimo rozsahu
     */
    public void setColorRgb(int red, int green, int blue)
    throws IllegalArgumentException;


    /**
     * Vráti x-ovú súradnicu vrcholu.
     * @return x-ová súradnica vrcholu.
     */
    public double getX();

    /**
     * Nastaví novú x-ovú súradnicu vrcholu. Súradnica má byť číslo od
     * 0 do 1, ak bude mimo intervalo, nastaví sa na 0 alebo 1.
     * @param x nová súradnica vrcholu.
     */
    public void setX(double x);

    /**
     * Vráti y-ovú súradnicu vrcholu.
     *
     * @return y-ová súradnica vrcholu.
     */
    public double getY();

    /**
     * Nastaví novú y-ovú súradnicu vrcholu. Súradnica má byť číslo od
     * 0 do 1, ak bude mimo intervalo, nastaví sa na 0 alebo 1.
     *
     * @param y nová súradnica vrcholu.
     */
    public void setY(double y);

}
