package graphgui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Collection;

/**
 * Trieda vrcholov grafu, implementácia rozhrania Vertex.
 * V svojej časti programu pristupujte ku grafu len cez
 * rozhrania Graph, Vertex a Edge.
 */
public final class VertexImplementation implements Vertex {

    private static final int DEFAULT_VALUE = 1;

    private static final String DEFAULT_COLOR_NAME = "white";

    private final GraphImplementation graph;
    private final ArrayList<EdgeImplementation> edges;
    private String colorName = DEFAULT_COLOR_NAME;
    private int id;
    private int value = DEFAULT_VALUE;
    @SuppressWarnings("checkstyle:MemberName")
    private double x;
    @SuppressWarnings("checkstyle:MemberName")
    private double y;

    VertexImplementation(GraphImplementation graph, int id, double x, double y) {
        edges = new ArrayList<>();
        this.id = id;
        this.graph = graph;
        this.x = fitCoordinate(x);
        this.y = fitCoordinate(y);
    }

    @Override
    public int getId() {
        return id;
    }

    void internalSetId(int id) {
        if (id != this.id) {
            this.id = id;
            graph.graphVertexChanged(this);
        }
    }

    @Override
    public Collection<Vertex> adjVertices() {
        ArrayList<Vertex> a = new ArrayList<>();
        for (Edge e : edges) {
            a.add(e.getOtherEnd(this));
        }
        return Collections.unmodifiableList(a);
    }

    @Override
    public Collection<Integer> adjVertexIds() {
        ArrayList<Integer> a = new ArrayList<>();
        for (Edge e : edges) {
            a.add(e.getOtherEnd(id));
        }
        return Collections.unmodifiableList(a);
    }

    @Override
    public Collection<Edge> adjEdges() {
        return Collections.unmodifiableList(new ArrayList<Edge>(edges));
    }

    void internalAddEdge(EdgeImplementation e) {
        edges.add(e);
    }

    boolean internalRemoveEdge(EdgeImplementation e) {
        return edges.remove(e);
    }

    @Override
    public EdgeImplementation findEdge(Vertex v) {
        for (EdgeImplementation e : edges) {
            if (e.isIncident(v)) {
                return e;
            }
        }
        return null;
    }

    @Override
    public void setValue(int value) {
        if (value != this.value) {
            this.value = value;
            graph.graphVertexChanged(this);
        }
    }

    @Override
    public int getValue() {
        return value;
    }

    private double fitCoordinate(double value) {
        if (value < 0.0) return 0.0;
        if (value > 1.0) return 1.0;
        return value;
    }

    @Override
    public void setX(double x) {
        x = fitCoordinate(x);
        if (x != this.x) {
            this.x = x;
            graph.graphVertexChanged(this);
        }
    }

    @Override
    public void setY(double y) {
        y = fitCoordinate(y);
        if (y != this.y) {
            this.y = y;
            graph.graphVertexChanged(this);
        }
    }

    @Override
    public void setColorRgb(int red, int green, int blue)
    throws IllegalArgumentException {
        String colorString
            = EdgeImplementation.colorNameFromRgb(red, green, blue);
        this.setColorName(colorString);
    }

    @Override
    public void setColorName(String colorName) {
        if (!this.colorName.equals(colorName)) {
            EdgeImplementation.verifyColorName(colorName);
            this.colorName = colorName;
            graph.graphVertexChanged(this);
        }
    }

    @Override
    public double getX() {
        return x;
    }

    @Override
    public double getY() {
        return y;
    }

    @Override
    public String getColorName() {
        return colorName;
    }

    @Override
    public String toString() {
        return String.format("%d (%2f,%2f) value %d", getId(), getX(), getY(), getValue());
    }
}
