#include <vector>
#include <algorithm>
#include <limits>
#include <cstdio>
#include "heuristic.h"

using namespace std;

class Solver {
private:
  unsigned n_;
  vector<vector<int>> distances_;
  int best_value_;
  vector<unsigned> best_vertex_order_;
  vector<unsigned> current_vertex_order_;
  AbstractHeuristic *heuristic_;
  
  
  static void CopyWithoutElement(const vector<unsigned> &source, vector<unsigned> &destination, unsigned skipped_index) {
    for (unsigned i = 0; i < skipped_index; i++) {
      destination[i] = source[i];
    }
    for (unsigned i = skipped_index + 1; i < source.size(); i++) {
      destination[i-1] = source[i];
    }
  }
  
  void SolveInternal(unsigned current_vertex, const vector<unsigned> &remaining_vertices, int current_value) {
    unsigned remaining_count = remaining_vertices.size();
    if (remaining_count == 0) {
      current_value += distances_[current_vertex][0];
      if (current_value < best_value_) {
        best_value_ = current_value;
        best_vertex_order_ = current_vertex_order_;
        fprintf(stderr, "Found new best route of cost %d:\n", best_value_);
        for (unsigned vertex : current_vertex_order_) {
          fprintf(stderr, "%d ", vertex);
        }
        fprintf(stderr, "\n");
      }
      return;
    }
    vector<pair<int, unsigned>> child_by_heuristic(remaining_count);
    vector<unsigned> child_remaining_vertices(remaining_count - 1);
    for (unsigned child_index = 0; child_index < remaining_count; child_index++) {
      CopyWithoutElement(remaining_vertices, child_remaining_vertices, child_index);
      unsigned child = remaining_vertices[child_index];
      int edge_length = distances_[current_vertex][child];
      int heuristic_value = heuristic_->LowerBound(child, 0, child_remaining_vertices, best_value_ - current_value - edge_length);
      int child_lower_bound = current_value + edge_length + heuristic_value;
      child_by_heuristic[child_index] = {child_lower_bound, child_index};
    }
    sort(child_by_heuristic.begin(), child_by_heuristic.end());
    for (const pair<int, unsigned> &child_pair : child_by_heuristic) {
      int lower_bound = child_pair.first;
      if (lower_bound >= best_value_) {
        break;
      }
      unsigned child_index = child_pair.second;
      unsigned child = remaining_vertices[child_index];
      current_vertex_order_.push_back(child);
      CopyWithoutElement(remaining_vertices, child_remaining_vertices, child_index);
      int edge_length = distances_[current_vertex][child];
      SolveInternal(child, child_remaining_vertices, current_value + edge_length);
      current_vertex_order_.pop_back();
    }
    return;
  }
  
public:
  Solver(unsigned n, vector<vector<int>> distances) 
    : n_(n),
      distances_(distances),
      best_value_(numeric_limits<int>::max()) {
    heuristic_ = CreateHeuristic(n, distances);
  }
  
  pair<int, vector<unsigned>> Solve() {
    if (best_value_ == numeric_limits<int>::max()) {
      vector<unsigned> remaining_vertices(n_-1);
      for (unsigned i = 0; i < n_-1; i++) {
        remaining_vertices[i] = i + 1;
      }
      current_vertex_order_.resize(1);
      current_vertex_order_[0] = 0;
      SolveInternal(0, remaining_vertices, 0);
    }
    return {best_value_, best_vertex_order_};
  }
  
  ~Solver() {
    delete heuristic_;
  }
};


int main() {
  unsigned n;
  scanf("%u", &n);
  vector<vector<int>> distances(n, vector<int> (n));
  for (unsigned from = 0; from < n; from++) {
    for (unsigned to = 0; to < n; to++) {
      scanf("%d", &distances[from][to]);
    }
  }
  Solver solver(n, distances);
  pair<int, vector<unsigned>> result = solver.Solve();
  printf("%d\n", result.first);
  for (unsigned i = 0; i < n; i++) {
    printf("%u%c", result.second[i], " \n"[i+1 == n]);
  }
  return 0;
}
