package shared;

use strict;
use POSIX;

BEGIN {
    use Exporter   ();
    our (@ISA, @EXPORT, @EXPORT_OK);
    @ISA = qw(Exporter);
    # symbols to export by default
    @EXPORT = qw(normalize_fasta_name read_fasta write_fasta 
		 my_run 
                 parse_options read_config adjust_relpath 
                 temp_dir_name $PARAM_PATH 
		 read_gtf parse_gtf_line write_gtf);
}

use FindBin qw($Bin);  #$Bin is the directory with the script

# Path to be rewritten by installing script - DO NOT CHANGE!!
our $PARAM_PATH = "$Bin/../param"; #REPLACE_PARAM_PATH 

############################
sub temp_dir_name {
    my ($query) = @_;
    # gets name of the fasta and creates a directory that 
    # reflacts this name. This is then used to store
    # temporary files of exonhunter.

    my $res = "ehtemp_" . $query;
    $res =~ s/\W+/-/g;
    return $res;
}

############################
sub read_config {
    my ($filename) = @_;
    # read a give filename, remove comments and empty lines, 
    # trim whitespace and return the result as a list of lines
    # !! to do: accept split lines with \ at the end.

    local *IN;
    open IN, "<$filename" or die "Cannot open $filename for reading";
    my @result;
    while(my $line = <IN>) {
	$line =~ s/\#.*$//;  #remove comments
	$line =~ s/\s+$//;   #trim
	$line =~ s/^\s+//;   #trim
	if($line ne '') {
	    push @result, $line;
	}
    }
    close IN or die;
    return @result;
}

############################
sub parse_options {
    my ($options, $lines) = @_;
    #parse lines into variable name and value, store in a hash

    foreach my $line (@$lines) {
	$line =~ /^([^=\s]+)\s*\=\s*(.*)$/ or die $line;
	$options->{$1} = $2;
    }
}

############################
sub my_run
{
    my ($run, $die) = @_;
    if(!defined($die)) { $die = 1; }

    my $short = substr($run, 0, 20);

    print STDERR $run, "\n";
    my $res = system("bash", "-c", $run);
    if($res<0) {
        die "Error in program '$short...' '$!'";
    }
    if($? && $die) {
        my $exit  = $? >> 8;
        my $signal  = $? & 127;
        my $core = $? & 128;

        die "Error in program '$short...' "
            . "(exit: $exit, signal: $signal, dumped: $core)\n\n ";
    }
}


############################
sub read_fasta {
    # Read one fasta sequence from the fasta file
    # Return undef, if no sequence found; 
    #        name and reference to sequence otherwise.
    
    my ($input) = @_;
    
    # read name of fasta sequence
    my $name;
    while(1) {
	my $line = <$input>;
	
	# end of file
	return unless defined $line;

	# skip empty lines
	next if $line =~ /^\s*$/;

	# parse the name
	$line =~ s/\s+$//;
	if($line =~ /^>/) {
	    $name = $line; 
	    last;
	}
	else { die "Incorrect fasta file '$line'."; }
    }

    # read fasta sequence
    my $sequence = "";
    while(1) {
	my $file_pos = tell($input);
	my $line = <$input>;
	
	# end of file
	last unless defined $line; 

	# if there is a beginning of a new sequence
	if($line =~ /^>/) {
	    seek($input, $file_pos, 0);
	    last;
	}

	# remove all whitespaces from line and append it to the sequence
	$line =~ s/\s//g;
	$sequence .= $line;
    }
    
    return ($name, \$sequence);
}

############################
sub read_gtf
{
    my ($in, $name) = @_;

    my @rows;

    while(1) {
	my $file_pos = tell $in;   # remember where the line starts
	
	my $line = <$in>;
	last if !defined($line);

	next if $line =~ /^\#/;    #skip comments
	$line =~ s/\s+$//;         #strip trailing whitespace
	next if $line eq '';       #skip empty lines

	my ($seq, $rest) = split "\t", $line, 2;
	if($seq ne $name) {
	    seek $in, $file_pos, SEEK_SET;   #return to the start of line
	    last; 
	}
	else {                     # add one more line to sequence
	    push @rows, $line;
	}
    }
    return (\@rows);
}

########################
sub parse_gtf_line
{
    my ($line) = @_;

    return if $line =~ /^\#/;    #skip comments
    $line =~ s/\s+$//;
    return if $line eq '';       #skip empty lines

    my %res;

    @res{'seqname', 'source', 'feature', 'start', 'end', 'score',
        'strand', 'frame', 'attributes'} = split "\t", $line;

    die "Bad gtf line '$line'\n " if(!defined $res{'frame'});

    if(defined($res{attributes})) {
	my @attr = split ";", $res{"attributes"};
	foreach my $attrib (@attr) {
	    if($attrib =~ /^\s*(\S+)\s+"(.*)"\s*$/) {
		if(!exists($res{$1})) {
		    $res{$1} = $2;
		}
	    }
	}
    }
    return \%res;
}

############################
sub write_gtf
{
    my ($output, $rows) = @_;

    my @gtf_col_keys = ("seqname", "source", "feature", "start",
                        "end", "score", "strand", "frame");
    my %gtf_mandatory_cols;
    foreach my $col (@gtf_col_keys) { $gtf_mandatory_cols{$col} = 1; }

    foreach my $row (@$rows) {

        my $sep = "";
        foreach my $col (@gtf_col_keys) {
	    die "Bad row (missing $col)", Dumper($row), " "
		unless defined $row->{$col};
	    print $output $sep, $row->{$col};
	    $sep = "\t";
        }

	# [attributes]
        foreach my $col (sort keys %$row) {
            if(!($col eq "attributes")
               && !(exists $gtf_mandatory_cols{$col})) {
                printf $output "%s%s \"%s\";", $sep,
                $col, $row->{$col};
                $sep = " ";
            }
        }
	print $output "\n";
    }
}

############################
sub normalize_fasta_name
{
    my ($name) = @_;

    $name =~ s/\s+$//;          # remove trailing white space
    $name =~ s/^>\s*//;         # remove > at the beginning of line
    $name =~ s/^(\S+)\s.*$/$1/; # remove everything after first space

    #if it has gi, take only number
    if ( $name =~ /gi\|(.*)\|/ ) {
        $name = $1;
    }

    return $name;
}

############################
sub write_fasta {
    my ($file, $name, $seq) = @_;

    print $file $name, "\n";
    my $n = length($$seq);
    my $linelen = 60;
    my $i=0;
    while($i<$n) {
        print $file (substr($$seq, $i, $linelen), "\n");
        $i+=$linelen;
    }
}

############################
sub adjust_relpath {
    #transform filename relative to current directory 
    #to be valid relative to its subdirectory

    #!! investigate File::Spec

    my ($path) = @_;

    #skip absolute paths
    return $path if($path =~ /^\//);
    return $path if($path =~ /^\~/);
    
    return "../$path";
}


1;
