import java.util.ArrayList;
import java.util.Arrays;

/**
 * Efficient Set vs Set DCJ distance computation
 */
public class DcjDistance implements EfficientDistance {
	
	private int geneCount;
	
	private int distance;
	private int[][] distances;
	
	private int componentCount;
	
	// extremity->component
	private int[] extremityLocation;
	
	// extremity->position in component
	private int[] extremityPosition;
	
	private ArrayList<Integer> componentSize;
	private ArrayList<Boolean> componentCircular;
	
	private CompositeComponent[] extremityTrack;
	
	private ArrayList<CompositeComponent> alteredInitialCC;
	
	public static void main(String[] args) {
		//test1();
		testManyvsMany(65, false);
	}
	
	private static void test1() {
		Genome genome1 = new Genome("1 3 -4 $ 5 2 @ 6 7 $");
		Genome genome2 = new Genome("2 1 @ 3 4 $ 5 $ 7 6 @");
		
		DcjDistance dcjDistance = new DcjDistance(genome1, genome2);
		System.out.println("distance: " + dcjDistance.calculateDistance());
	}
	
	private static boolean testManyvsMany(int geneCount, boolean save) {
		Genome genome1 = new Genome(Genome.generateRandomGenomeString(geneCount));
		Genome genome2 = new Genome(Genome.generateRandomGenomeString(geneCount));
		//Genome genome1 = new Genome("3 $ 1 @ 2 @ 4 @");
		//Genome genome2 = new Genome("1 -3 -2 @ 4 @");
		Neighbours neigh1 = neighbourGenerator(genome1);
		Neighbours neigh2 = neighbourGenerator(genome2);
		
		System.out.println("neighbourhood of " + genome1.getStringRepresentation() + " vs neighbourhood of " + genome2.getStringRepresentation());
		
		long s1 = System.currentTimeMillis();
		DcjDistance dcjDistance = new DcjDistance(genome1, genome2);
		int distances[][] = dcjDistance.distance(neigh1.breaks, neigh1.joins, neigh2.breaks, neigh2.joins, save);
		long e1 = System.currentTimeMillis();
		System.out.println("effective: " + (e1-s1));
		
		DCJ dcj=new DCJ();
		boolean ok = true;
		long s2=System.currentTimeMillis();
		for (int i=0; i<neigh1.size(); i++) {
			for (int j=0; j<neigh2.size(); j++) {
				int correctDist = dcj.distance(neigh1.neighbours[i], neigh2.neighbours[j]);
//				if (save && distances[i][j] != correctDist) {
//					System.out.println("I="+i+" J="+j+"  " + distances[i][j] + " vs " + correctDist + "  BREAKS1: " + Arrays.toString(neigh1.breaks[i]) + " JOINS1: " + Arrays.toString(neigh1.joins[i]) + "   BREAKS2: " + Arrays.toString(neigh2.breaks[j]) + " JOINS:" + Arrays.toString(neigh2.joins[j]));
//					ok = false;
//				}
			}
		}
		long e2 = System.currentTimeMillis();
		
		if (ok) System.out.println("OK");
		else System.out.println("ERROR");
		System.out.println("effective: " + (e1-s1) + " vs bruteforce: " + (e2-s2));
		
		return ok;
	}
	
	private static Neighbours neighbourGenerator(Genome genome) {
		Genome[] neighbours = new Genome[genome.getGeneCount()*2*(genome.getGeneCount()*2-1)];
		int[][] breaks = new int[genome.getGeneCount()*2*(genome.getGeneCount()*2-1)][];
		int[][] joins = new int[genome.getGeneCount()*2*(genome.getGeneCount()*2-1)][];
		DCJ dcj = new DCJ();
		
		int nc = 0;
		for (int i=0; i<genome.getGeneCount()*2; i++) {
			for (int j=0; j<genome.getGeneCount()*2; j++) {
				if (i==j) continue;
				Genome mutated = new Genome(genome);
				int type = dcj.dcjOperation(mutated, i, j);
				neighbours[nc] = mutated;
				
				int extA = Genome.getExtremityByAdjacencyIndex(i);
				int extB = genome.getAdjacentExtremityByIndex(i);
				int extC = Genome.getExtremityByAdjacencyIndex(j);
				int extD = genome.getAdjacentExtremityByIndex(j);
				
				switch (type) {
				case 1:
					// nothing is broken, only join
					breaks[nc] = new int[0];
					joins[nc] = new int[2];
					joins[nc][0] = extA; joins[nc][1] = extC;
					break;
				case 2:
					breaks[nc] = new int[2];
					breaks[nc][0] = extA; breaks[nc][1] = extB; 
					joins[nc] = new int[2];
					joins[nc][0] = extA; joins[nc][1] = extC;
					break;
				case 3:
					breaks[nc] = new int[2];
					breaks[nc][0] = extC; breaks[nc][1] = extD; 
					joins[nc] = new int[2];
					joins[nc][0] = extA; joins[nc][1] = extC;
					break;
				case 4:
					if (extA==extD && extB==extC) {
						// no joins, only break
						breaks[nc] = new int[2];
						breaks[nc][0] = extA; breaks[nc][1] = extB; 
						joins[nc] = new int[0];
					}
					else {
						breaks[nc] = new int[4];
						breaks[nc][0] = extA; breaks[nc][1] = extB; breaks[nc][2] = extC; breaks[nc][3] = extD; 
						joins[nc] = new int[4];
						joins[nc][0] = extA; joins[nc][1] = extD; joins[nc][2] = extB; joins[nc][3] = extC;
					}
					break;
				}
				nc++;
			}
		}
		Neighbours neigh = new Neighbours();
		neigh.neighbours = neighbours;
		neigh.breaks = breaks;
		neigh.joins = joins;
		return neigh;
	}
	
	public DcjDistance(Genome genome1, Genome genome2) {
		if (genome1.getGeneCount()!=genome2.getGeneCount()) throw new RuntimeException("not equal gene content");
		geneCount = genome1.getGeneCount();

		componentCount = 0;
		
		// extremity->component
		extremityLocation = new int[geneCount*2 + geneCount*2];
		
		// extremity->position in component
		extremityPosition = new int[geneCount*2 + geneCount*2];
		
		componentSize = new ArrayList<>();
		componentCircular = new ArrayList<>();
		
		calculateComponents(genome1, genome2);
		
		calculateDistance();
		
//		System.out.println("gene count: " + geneCount);
//		System.out.println("component count: " + componentCount);
//		System.out.println("extremity location: " + Arrays.toString(extremityLocation));
//		System.out.println("extremity position: " + Arrays.toString(extremityPosition));
//		System.out.println("component size: " + componentSize);
//		System.out.println("component circular: " + componentCircular);
		
		initializeCompositeComponents();
	}

	private void calculateComponents(Genome genome1, Genome genome2) {
		boolean adj1[] = new boolean[geneCount*2];
		for (int i = 0; i < adj1.length; i++) {
			adj1[i] = false;
		}
		boolean adj2[] = new boolean[geneCount*2];
		for (int i = 0; i < adj2.length; i++) {
			adj2[i] = false;
		}
		
		for (int i = 0; i < adj1.length; i++) {
			if (adj1[i]==true) continue;
			if (genome1.isIndexTelomere(i)) {
				int size = analyzeComponent(genome1, genome2, adj1, adj2, 0, Genome.getExtremityByAdjacencyIndex(i));
				componentSize.add(size);
				componentCircular.add(false);
			}
		}
		for (int i = 0; i < adj2.length; i++) {
			if (adj2[i]==true) continue;
			if (genome2.isIndexTelomere(i)) {
				int size = analyzeComponent(genome1, genome2, adj1, adj2, 1, Genome.getExtremityByAdjacencyIndex(i));
				componentSize.add(size);
				componentCircular.add(false);
			}
		}
		for (int i = 0; i < adj1.length; i++) {
			if (adj1[i]==true) continue;
				int size = analyzeComponent(genome1, genome2, adj1, adj2, 0, Genome.getExtremityByAdjacencyIndex(i));
				componentSize.add(size);
				componentCircular.add(true);
		}
	}
	
	private int calculateDistance() {
		distance = geneCount;
		int oddPaths = 0;
		for(int i=0; i<componentCount; i++) {
			if (componentCircular.get(i)) distance--;
			else if ((componentSize.get(i)/2)%2==1) oddPaths++; 
		}
		distance-=oddPaths/2;
		return distance;
	}

	private int analyzeComponent(Genome genome1, Genome genome2,
			boolean[] adj1, boolean[] adj2, int parity, int extremity) {
		componentCount++;
		int length = parity;
		int position = 0;
		while (true) {
			int adjacencyIndex = Genome.getAdjacencyIndexByExtremity(extremity);
			
			// we closed a cycle
			if (length%2==0 && adj1[adjacencyIndex]) break;
			if (length%2==1 && adj2[adjacencyIndex]) break;
			
			// mark as visited
			if (length%2==0) {
				adj1[adjacencyIndex] = true;
				extremityLocation[adjacencyIndex] = componentCount-1;
				extremityPosition[adjacencyIndex] = position;
				//System.out.println(position + ". " + adjacencyIndex + " ext:" + Genome.getExtremityByAdjacencyIndex(adjacencyIndex) + " (" + (componentCount-1) + ")");
				position++;
			}
			if (length%2==1) {
				adj2[adjacencyIndex] = true;
				extremityLocation[adjacencyIndex+geneCount*2] = componentCount-1;
				extremityPosition[adjacencyIndex+geneCount*2] = position;
				//System.out.println(position + ". " + (adjacencyIndex+geneCount*2) + " ext:" + Genome.getExtremityByAdjacencyIndex(adjacencyIndex) + " (" + (componentCount-1) + ")");
				position++;
			}
			
			// maybe we are at the end of a path
			if (length-parity>0) {
				if (length%2==0 && genome1.isExtremityTelomere(extremity)) break;
				if (length%2==1 && genome2.isExtremityTelomere(extremity)) break;
			}
			
			// ok we are not... mark the other extremity in the adjacency ... only if we are not in the beggining of a path
			if (length%2==0 && !genome1.isExtremityTelomere(extremity)) {
				int otherAdjacencyIndex=Genome.getAdjacencyIndexByExtremity(genome1.getAdjacentExtremityByExtremity(extremity));
				adj1[otherAdjacencyIndex] = true;
				extremityLocation[otherAdjacencyIndex] = componentCount-1;
				extremityPosition[otherAdjacencyIndex] = position;
				//System.out.println(position + ". " + otherAdjacencyIndex + " ext:" + Genome.getExtremityByAdjacencyIndex(otherAdjacencyIndex) + " (" + (componentCount-1) + ")");
				position++;
			}
			if (length%2==1 && !genome2.isExtremityTelomere(extremity)) {
				int otherAdjacencyIndex=Genome.getAdjacencyIndexByExtremity(genome2.getAdjacentExtremityByExtremity(extremity));
				adj2[otherAdjacencyIndex] = true;
				extremityLocation[otherAdjacencyIndex+geneCount*2] = componentCount-1;
				extremityPosition[otherAdjacencyIndex+geneCount*2] = position;
				//System.out.println(position + ". " + (otherAdjacencyIndex+geneCount*2) + " ext:" + Genome.getExtremityByAdjacencyIndex(otherAdjacencyIndex) + " (" + (componentCount-1) + ")");
				position++;
			}

			// select the next extremity
			if (length%2==0) extremity = genome1.getAdjacentExtremityByIndex(adjacencyIndex);
			if (length%2==1) extremity = genome2.getAdjacentExtremityByIndex(adjacencyIndex);
			
			// increase length
			length++;
		}

		return position;
	}
	
	private void initializeCompositeComponents() {
		CompositeComponent[] initialCC = new CompositeComponent[componentCount];
		for (int i=0; i<componentCount; i++) {
			ArrayList<CompositeComponentContent> contents = new ArrayList<>();
			contents.add(new CompositeComponentContent(i, 0, componentSize.get(i)-1, false));
			initialCC[i] = new CompositeComponent(contents, componentCircular.get(i), true);
		}
		extremityTrack = new CompositeComponent[geneCount*2*2];
		for (int i=0; i<extremityTrack.length; i++) {
			extremityTrack[i] = initialCC[extremityLocation[i]];
		}
		alteredInitialCC = new ArrayList<>();
	}

	public int[][] distance(int[][] break1, int[][] join1, int[][] break2, int[][] join2, boolean save) {
		if (save) distances = new int[break1.length][break2.length];
		else distances = new int[0][0];
		
		for(int i=0; i<break1.length; i++) {
			for(int j=0; j<break2.length; j++) {
				int distanceChange = 0;
				//System.out.println(Arrays.toString(break1[i])+Arrays.toString(join1[i])+Arrays.toString(break2[j])+Arrays.toString(join2[j]));
				for(int d=0; d<break1[i].length; d+=2) {
					int ch = trackBreak(break1[i][d], break1[i][d+1]);
					//System.out.print(ch + " ");
					distanceChange += ch;
				}
				for(int d=0; d<join1[i].length; d+=2) {
					int ch = trackJoin(join1[i][d], join1[i][d+1]);
					//System.out.print(ch + " ");
					distanceChange += ch;
				}
				for(int d=0; d<break2[j].length; d+=2) {
					int ch = trackBreak(break2[j][d] + Integer.signum(break2[j][d])*geneCount, break2[j][d+1] + Integer.signum(break2[j][d+1])*geneCount);
					//System.out.print(ch + " ");
					distanceChange += ch;
				}
				for(int d=0; d<join2[j].length; d+=2) {
					int ch = trackJoin(join2[j][d] + Integer.signum(join2[j][d])*geneCount, join2[j][d+1] + Integer.signum(join2[j][d+1])*geneCount);
					//System.out.print(ch + " ");
					distanceChange += ch;
				}
				//System.out.println();
				if (save) distances[i][j] = distance+distanceChange;
				clearTrack();
			}
		}
		
		return distances;
	}
	
	public int distance(int[] break1, int[] join1, int[] break2, int[] join2) {
		int distanceChange = 0;
		//System.out.println(Arrays.toString(break1[i])+Arrays.toString(join1[i])+Arrays.toString(break2[j])+Arrays.toString(join2[j]));
		for(int d=0; d<break1.length; d+=2) {
			int ch = trackBreak(break1[d], break1[d+1]);
			//System.out.print(ch + " ");
			distanceChange += ch;
		}
		for(int d=0; d<join1.length; d+=2) {
			int ch = trackJoin(join1[d], join1[d+1]);
			//System.out.print(ch + " ");
			distanceChange += ch;
		}
		for(int d=0; d<break2.length; d+=2) {
			int ch = trackBreak(break2[d] + Integer.signum(break2[d])*geneCount, break2[d+1] + Integer.signum(break2[d+1])*geneCount);
			//System.out.print(ch + " ");
			distanceChange += ch;
		}
		for(int d=0; d<join2.length; d+=2) {
			int ch = trackJoin(join2[d] + Integer.signum(join2[d])*geneCount, join2[d+1] + Integer.signum(join2[d+1])*geneCount);
			//System.out.print(ch + " ");
			distanceChange += ch;
		}
		//System.out.println();
		int answer = distance+distanceChange;
		clearTrack();

		return answer;
	}

	private int trackBreak(int ext1, int ext2) {
		CompositeComponent CCext1 = findCompositeComponent(ext1);
		CompositeComponent CCext2 = findCompositeComponent(ext2);
		if (CCext1!=CCext2) throw new RuntimeException("error");
		CompositeComponent CC = CCext1; // they have to be in same CompositeComponent
		
		int contentIndex1 = CC.find(ext1);	// in which content is ext1
		int contentIndex2 = CC.find(ext2);  // in which content is ext2
		int position1 = extremityPosition[getIndexFromExtremity(ext1)];
		int position2 = extremityPosition[getIndexFromExtremity(ext2)];
		
		if (CC.isLinear()) {
			ArrayList<CompositeComponentContent> leftContents = new ArrayList<>();
			ArrayList<CompositeComponentContent> rightContents = new ArrayList<>();
			for (int i=0; i<Math.max(contentIndex1, contentIndex2); i++) {
				leftContents.add(new CompositeComponentContent(CC.contents.get(i).component, CC.contents.get(i).start, CC.contents.get(i).end, CC.contents.get(i).direction));
			}
			if (contentIndex1==contentIndex2) {
				if (CC.contents.get(contentIndex1).direction==false) {
					leftContents.add(new CompositeComponentContent(CC.contents.get(contentIndex1).component, CC.contents.get(contentIndex1).start, Math.min(position1, position2), CC.contents.get(contentIndex1).direction));
					rightContents.add(new CompositeComponentContent(CC.contents.get(contentIndex1).component, Math.max(position1, position2), CC.contents.get(contentIndex1).end, CC.contents.get(contentIndex1).direction));
				}
				else {
					leftContents.add(new CompositeComponentContent(CC.contents.get(contentIndex1).component, Math.max(position1, position2),  CC.contents.get(contentIndex1).end, CC.contents.get(contentIndex1).direction));
					rightContents.add(new CompositeComponentContent(CC.contents.get(contentIndex1).component, CC.contents.get(contentIndex1).start, Math.min(position1, position2), CC.contents.get(contentIndex1).direction));
				}
			}
			for (int i=Math.min(contentIndex1, contentIndex2)+1; i<CC.contents.size(); i++) {
				rightContents.add(new CompositeComponentContent(CC.contents.get(i).component, CC.contents.get(i).start, CC.contents.get(i).end, CC.contents.get(i).direction));
			}
			
			CompositeComponent CCleft = new CompositeComponent(leftContents, false, false);
			CompositeComponent CCright = new CompositeComponent(rightContents, false, false);
			CC.addDescendant(CCleft);
			CC.addDescendant(CCright);
			if (CC.isInitial) {
				alteredInitialCC.add(CC);
			}
			
			// if CC is odd path, we do not need to do anything
			// if CC is even path, we have to check if CCleft and CCrigth are odd paths
			if ((CC.size()/2)%2==0) {
				if ((CCleft.size()/2)%2==1) return -1;
			}
			return 0;
		}
		else {
			// circular
			ArrayList<CompositeComponentContent> newContents = new ArrayList<>();
			if (contentIndex1==contentIndex2) {
				if (Math.abs(position1-position2)!=1) throw new RuntimeException();
				if (CC.contents.get(contentIndex1).direction==false) {
					newContents.add(new CompositeComponentContent(CC.contents.get(contentIndex1).component, Math.max(position1, position2), CC.contents.get(contentIndex1).end, CC.contents.get(contentIndex1).direction));
				}
				else {
					newContents.add(new CompositeComponentContent(CC.contents.get(contentIndex1).component, CC.contents.get(contentIndex1).start, Math.min(position1, position2), CC.contents.get(contentIndex1).direction));
				}
			}
			for (int i=Math.min(contentIndex1, contentIndex2)+1; i<CC.contents.size(); i++) {
				newContents.add(new CompositeComponentContent(CC.contents.get(i).component, CC.contents.get(i).start, CC.contents.get(i).end, CC.contents.get(i).direction));
			}
			for (int i=0; i<Math.max(contentIndex1, contentIndex2); i++) {
				newContents.add(new CompositeComponentContent(CC.contents.get(i).component, CC.contents.get(i).start, CC.contents.get(i).end, CC.contents.get(i).direction));
			}
			if (contentIndex1==contentIndex2) {
				if (CC.contents.get(contentIndex1).direction==false) {
					newContents.add(new CompositeComponentContent(CC.contents.get(contentIndex1).component, CC.contents.get(contentIndex1).start, Math.min(position1, position2), CC.contents.get(contentIndex1).direction));
				}
				else {
					newContents.add(new CompositeComponentContent(CC.contents.get(contentIndex1).component, Math.max(position1, position2),  CC.contents.get(contentIndex1).end, CC.contents.get(contentIndex1).direction));
				}
			}
			
			CompositeComponent CCnew = new CompositeComponent(newContents, false, false);
			CC.addDescendant(CCnew);
			if (CC.isInitial) {
				alteredInitialCC.add(CC);
			}
			
			// we have destroyed one cycle and created a even path
			return 1;
		}
	}
	
	private int trackJoin(int ext1, int ext2) {
		CompositeComponent CCext1 = findCompositeComponent(ext1);
		CompositeComponent CCext2 = findCompositeComponent(ext2);
		
		int contentIndex1 = CCext1.find(ext1);	// in which content is ext1
		int contentIndex2 = CCext2.find(ext2);  // in which content is ext1
		int position1 = extremityPosition[getIndexFromExtremity(ext1)];
		int position2 = extremityPosition[getIndexFromExtremity(ext2)];
		
		if (CCext1==CCext2) {
			CompositeComponent CC = CCext1;
			ArrayList<CompositeComponentContent> newContents = new ArrayList<>();
			for (int i=0; i<CC.contents.size(); i++) newContents.add(new CompositeComponentContent(CC.contents.get(i).component, CC.contents.get(i).start, CC.contents.get(i).end, CC.contents.get(i).direction));
			CompositeComponent CCnew = new CompositeComponent(newContents, true, false);
			CC.addDescendant(CCnew);
			if (CC.isInitial) {
				alteredInitialCC.add(CC);
			}
			
			// -> new cycle
			return -1;
		}
		else {
			// CCext1!=CCext2
			ArrayList<CompositeComponentContent> newContents = new ArrayList<>();
			
			// ext1 is at the beggining
			if ((contentIndex1==0 && CCext1.contents.get(contentIndex1).direction==false && position1==CCext1.contents.get(contentIndex1).start) ||
					(contentIndex1==0 && CCext1.contents.get(contentIndex1).direction==true && position1==CCext1.contents.get(contentIndex1).end)) {
				for (int i=CCext1.contents.size()-1; i>=0; i--) {
					newContents.add(new CompositeComponentContent(CCext1.contents.get(i).component, CCext1.contents.get(i).start, CCext1.contents.get(i).end, !CCext1.contents.get(i).direction));
				}
			}
			else {
				// ext1 is at the end
				for (int i=0; i<CCext1.contents.size(); i++) {
					newContents.add(new CompositeComponentContent(CCext1.contents.get(i).component, CCext1.contents.get(i).start, CCext1.contents.get(i).end, CCext1.contents.get(i).direction));
				}
			}
			
			// ext2 is at the beggining
			if ((contentIndex2==0 && CCext2.contents.get(contentIndex2).direction==false && position2==CCext2.contents.get(contentIndex2).start) ||
					(contentIndex2==0 && CCext2.contents.get(contentIndex2).direction==true && position2==CCext2.contents.get(contentIndex2).end)) {
				for (int i=0; i<CCext2.contents.size(); i++) {
					newContents.add(new CompositeComponentContent(CCext2.contents.get(i).component, CCext2.contents.get(i).start, CCext2.contents.get(i).end, CCext2.contents.get(i).direction));
				}
			}
			else {
				// ext2 is at the end
				for (int i=CCext2.contents.size()-1; i>=0; i--) {
					newContents.add(new CompositeComponentContent(CCext2.contents.get(i).component, CCext2.contents.get(i).start, CCext2.contents.get(i).end, !CCext2.contents.get(i).direction));
				}
			}
			
			CompositeComponent CCnew = new CompositeComponent(newContents, false, false);
			CCext1.addDescendant(CCnew);
			CCext2.addDescendant(CCnew);
			if (CCext1.isInitial) {
				alteredInitialCC.add(CCext1);
			}
			if (CCext2.isInitial) {
				alteredInitialCC.add(CCext2);
			}
			
			// we have joined two paths... the distance changes only if both were odd
			if (((CCext1.size()/2)%2==1) && ((CCext2.size()/2)%2==1)) {
				return 1;
			}
			return 0;
		}
	}
	
	private CompositeComponent findCompositeComponent(int ext) {
		CompositeComponent CC = extremityTrack[getIndexFromExtremity(ext)];
		while (CC.hasDescendants()) {
			for (int i=0; i<CC.descendants.length; i++) {
				if (CC.descendants[i]!=null && CC.descendants[i].find(ext)!=-1) {
					CC = CC.descendants[i];
					break;
				}
			}
		}
		return CC;
	}


	private void clearTrack() {
		for (int i=0; i<alteredInitialCC.size(); i++) {
			alteredInitialCC.get(i).descendants[0]=null;
			alteredInitialCC.get(i).descendants[1]=null;
		}
		alteredInitialCC.clear();
	}
	
	private int getIndexFromExtremity(int ext) {
		int index=0;
		if (Math.abs(ext)>geneCount) {
			index+=geneCount*2;
			ext = ext-Integer.signum(ext)*geneCount;
		}
		index+=Genome.getAdjacencyIndexByExtremity(ext);
		return index;
	}

	class CompositeComponent {
		ArrayList<CompositeComponentContent> contents;
		
		boolean circular;
		
		CompositeComponent[] descendants;
		boolean isInitial;
		
		public CompositeComponent(ArrayList<CompositeComponentContent> contents, boolean circular, boolean isInitial) {
			this.contents = contents;
			this.circular = circular;
			this.descendants = new CompositeComponent[2];
			this.isInitial = isInitial;
		}
		
		public int find(int ext) {
			int location = extremityLocation[getIndexFromExtremity(ext)];
			int position = extremityPosition[getIndexFromExtremity(ext)];
			for (int i=0; i<contents.size(); i++) {
				if (contents.get(i).component == location) {
					if (contents.get(i).start<=position && position<=contents.get(i).end) return i;
				}
			}
			return -1;
		}
		
		public boolean isLinear() {
			if (circular==false) return true;
			return false;
		}
		
		public void addDescendant(CompositeComponent CC) {
			if (descendants[0]==null) descendants[0]=CC;
			else if (descendants[1]==null) descendants[1]=CC;
			else throw new RuntimeException("error");
		}
		
		public boolean hasDescendants() {
			if (descendants[0]!=null) return true;
			return false;
		}
		
		public int size() {
			int size = 0;
			for (int i=0; i<contents.size(); i++) {
				size += Math.abs(contents.get(i).end-contents.get(i).start)+1;
			}
			return size;
		}
	}
	
	class CompositeComponentContent {
		int component;
		int start;
		int end;
		boolean direction; // false: normal, true: reversed
		
		public CompositeComponentContent(int component, int start, int end, boolean direction) {
			this.component = component;
			this.start = start;
			this.end = end;
			this.direction = direction;
		}
	}
	
	static class Neighbours {
		public Genome[] neighbours;
		public int[][] breaks;
		public int[][] joins;
		
		public int size() {
			return neighbours.length;
		}
	}
}
