import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Arrays;

/**
 * The PIVO2
 */
public class PIVO2 implements SmallPhylogenyResultHandler {
	private String jobName;
	private EvolutionModel model;

	/**
	 * Use this to run the PIVO2
	 * 
	 * Command-line arguments: 
	 * - phylogeny file name
	 * - extant genomes file name
	 * - job name
	 * - initialization method: "simple" or "random" (simple should be used)
	 * - number of rounds
	 * - neighbour generation method: "neighbour_all", "neighbour_2", "neighbour_1", "neighbour_0"
	 * - preferred chromosome type: "chromosome_all", "chromosome_l", "chromosome_c", "chromosome_lc"
	 * - joining solutions: "join_NUMBER", example: "join_0" or "join_15"
	 * - plateau: "plateau_NUMBER", use "plateau_0"
	 * - tabu search: "tabu" or "notabu"
	 * - efficient distance calculation: "efficient" or "noefficient"
	 * 
	 * Example arguments:
	 * phylogeny.txt leafgenomes.txt test01 simple 100 neighbour_all chromosome_all join_0 plateau_0 notabu efficient 
	 */
	public static void main(String[] args) throws FileNotFoundException, IOException {
		long time = System.currentTimeMillis();
		
		// read arguments
		String inputTree = args[0];
		String inputGenomes = args[1];
		String jobName = args[2];
		String searchMethod = args[3];
		int numberOfRounds = Integer.parseInt(args[4]);
		String neighbourGeneration = args[5];
		String chromosomeType = args[6];
		String joinSolutions = args[7];
		String plateau = args[8];
		String tabu = args[9];
		String efficient = args[10];

		Phylogeny phylogeny = new Phylogeny();
		phylogeny.parseTree(new FileReader(inputTree));
		phylogeny.loadGenomes(new FileReader(inputGenomes));
		
		DCJ dcj = new DCJ();
		if (neighbourGeneration.equals("neighbour_0")) dcj.setNeighbourGeneration(0);
		if (neighbourGeneration.equals("neighbour_1")) dcj.setNeighbourGeneration(1);
		if (neighbourGeneration.equals("neighbour_2")) dcj.setNeighbourGeneration(2);
		if (neighbourGeneration.equals("neighbour_all")) dcj.setNeighbourGeneration(3);
		
		int joinSolutionsCount = Integer.parseInt(joinSolutions.substring(joinSolutions.indexOf("_")+1));
		if (joinSolutionsCount>0) dcj.setJSCGenerator(new JoinSolutionsCandidateGenerator(joinSolutionsCount, jobName, dcj));

		SmallPhylogeny smallPhylogeny = new SmallPhylogeny(dcj, phylogeny);
		SmallPhylogenyResultHandler handler = new PIVO2(jobName, dcj);
		
		if (chromosomeType.equals("chromosome_l")) smallPhylogeny.setChromosomePenalty(new ChromosomePenalty_L());
		if (chromosomeType.equals("chromosome_c")) smallPhylogeny.setChromosomePenalty(new ChromosomePenalty_C());
		if (chromosomeType.equals("chromosome_lc")) smallPhylogeny.setChromosomePenalty(new ChromosomePenalty_LC());
		if (chromosomeType.equals("chromosome_all")) smallPhylogeny.setChromosomePenalty(new ChromosomePenalty_all());
		
		int plateauWalk = Integer.parseInt(plateau.substring(plateau.indexOf("_")+1));
		if (plateauWalk>0) smallPhylogeny.setPlateauWalk(plateauWalk);
		
		if (tabu.equals("tabu")) {
			TabuSearch tabuSearch = new TabuSearch();
			tabuSearch.setPenalty((float) 1/(phylogeny.getNodes().size()+1));
			smallPhylogeny.setTabuSearch(tabuSearch);
		}
		
		if (efficient.equals("efficient")) {
			smallPhylogeny.setUseEfficientDistance(true);
		}
		
		printArgs(jobName, time, args);
		
		if (searchMethod.equals("random")) smallPhylogeny.randomizedSearch(numberOfRounds, handler);
		else if (searchMethod.equals("simple")) smallPhylogeny.simpleSearch(numberOfRounds, handler);
		
		printLog(jobName, time, smallPhylogeny.getLog());
	}

	private static void printArgs(String name, long time, String[] args) {
		File file = new File(name+"_"+time+".txt");
		PrintWriter out;
		try {
			out = new PrintWriter(file);
			out.println("PIVO args:");
			for (String arg : args) {
				out.print(arg + " ");
			}
			out.println();
			out.close();
		} catch (FileNotFoundException e) {
			System.out.println("can't save config args");
			e.printStackTrace();
		}
	}
	
	private static void printLog(String name, long time, String log) {
		File logFile = new File(name+"_"+time+"_log.txt");
		PrintWriter writer;
		try {
			writer = new PrintWriter(logFile);
			writer.print(log);
			writer.close();
		} catch (FileNotFoundException e) {
			System.out.println("can't save log");
			e.printStackTrace();
		}
		
	}

	public PIVO2(String jobName, EvolutionModel model) {
		this.jobName = jobName;
		this.model = model;
	}

	@Override
	public void smallPhylogenyResult(float score, Phylogeny phylogeny) {
		try {
			int scoreWithoutPenalty = phylogeny.getScore(model);
			System.out.println("Score WO penalty: " + scoreWithoutPenalty);
			File file = new File(this.jobName+"/result_"+scoreWithoutPenalty+"_"+(Long.MAX_VALUE-System.currentTimeMillis())+".txt");
			file.getParentFile().mkdirs();
			PrintWriter out = new PrintWriter(file);
			out.print(phylogeny.getResult(new DCJ()));
			out.close();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}

}
