import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

/**
 * Phylogenetic tree
 */
public class Phylogeny {
	
	/**
	 * The root of the phylogenetic tree.
	 */
	private PhylogenyNode root;
	
	/**
	 * Every node of the phylogenetic tree (internal nodes and leaves).
	 */
	private List<PhylogenyNode> nodes = new ArrayList<PhylogenyNode>();
	
	/**
	 * The internal nodes of the phylogenetic tree.
	 */
	private List<PhylogenyNode> internalNodes = new ArrayList<PhylogenyNode>();
	
	/**
	 * The leaves of the phylogenetic tree.
	 */
	private List<PhylogenyNode> leaves = new ArrayList<PhylogenyNode>();
	
	/**
	 * Set of genes in the phylogenetic tree.
	 */
	private int geneCount = 0;
	
	public PhylogenyNode getRoot() {
		return root;
	}
	
	public List<PhylogenyNode> getNodes() {
		return nodes;
	}
	
	public List<PhylogenyNode> getInternalNodes() {
		return internalNodes;
	}
	
	public List<PhylogenyNode> getLeaves() {
		return leaves;
	}
	
	public int getGeneCount() {
		return geneCount;
	}
	
	public void setGeneCount(int geneCount) {
		this.geneCount = geneCount;
	}
	
	/**
	 * Constructor.
	 */
	public Phylogeny() {
		this.nodes = new ArrayList<PhylogenyNode>();
		this.internalNodes = new ArrayList<PhylogenyNode>();
		this.leaves = new ArrayList<PhylogenyNode>();
		this.root = null;
		this.geneCount = 0;
	}
	
	/**
	 * Copy constructor.
	 * Creates a deep copy of the Phylogeny object.
	 * @param phylogeny
	 */
	public Phylogeny(Phylogeny phylogeny) {
		this();
		this.geneCount = phylogeny.geneCount;
		this.root = copyPhylogenyNode(phylogeny.root, null);
	}
	
	private PhylogenyNode copyPhylogenyNode(PhylogenyNode phylogenyNode, PhylogenyNode parent) {
		PhylogenyNode newNode = new PhylogenyNode();
		
		newNode.setParent(parent);
		if (phylogenyNode.getLeft()!=null) newNode.setLeft(copyPhylogenyNode(phylogenyNode.getLeft(), newNode));
		if (phylogenyNode.getRight()!=null) newNode.setRight(copyPhylogenyNode(phylogenyNode.getRight(), newNode));
		if (phylogenyNode.isLeaf()) newNode.setLeaf(true);
		this.nodes.add(newNode);
		if (newNode.isLeaf()) this.leaves.add(newNode);
		else this.internalNodes.add(newNode);
		
		newNode.setOrganismName(phylogenyNode.getOrganismName());
		
		if (phylogenyNode.getGenome()!=null) newNode.setGenome(new Genome(phylogenyNode.getGenome()));
		
		if (phylogenyNode.isLeaf()) {
			for (Genome candidate : phylogenyNode.getCandidates()) {
				newNode.addCandidate(new Genome(candidate));
			}
		}
		
		// Note: candidates for other nodes, and other values are not copied
		
		return newNode;
	}
	
	/**
	 * Creates the structure of the phylogenetic tree
	 * <p>The root of the tree will be saved into {@link Phylogeny#root} 
	 * @param treeData
	 * @throws IOException 
	 * @see Phylogeny#root
	 * @see PhylogenyNode
	 */
	public void parseTree(Reader treeData) throws IOException {
		this.nodes = new ArrayList<PhylogenyNode>();
		this.internalNodes = new ArrayList<PhylogenyNode>();
		this.leaves = new ArrayList<PhylogenyNode>();
		this.root = null;
		
		String historyString = new BufferedReader(treeData).readLine();
		//String[] parts = historyString.replaceAll("[(:0123456789.;\n]", "").replace(",", " ").replace(")", " * ").split(" +");
		String[] parts = historyString.replaceAll(":[0123456789](\\.[0123456789]*)?", "").replaceAll("[(:.;\n]", "").replace(",", " ").replace(")", " * ").split(" +");
		Stack<PhylogenyNode> nodeStack = new Stack<PhylogenyNode>();
		for (String part : parts) {
			PhylogenyNode node = new PhylogenyNode();
			if (part.equals("*")) {
				PhylogenyNode right = nodeStack.pop();
				PhylogenyNode left = nodeStack.pop();
				node.setLeft(left);
				node.setRight(right);
				left.setParent(node);
				right.setParent(node);
				nodeStack.push(node);
				this.internalNodes.add(node);
			}
			else {
				node.setOrganismName(part);
				node.setLeaf(true);
				nodeStack.push(node);
				this.leaves.add(node);
			}
			this.nodes.add(node);
		}
		this.root = nodeStack.pop();
		
		createInternalNodeNames(this.root);
	}
	
	private void createInternalNodeNames(PhylogenyNode node) {
		if (node.isLeaf()) return;
		createInternalNodeNames(node.getLeft());
		createInternalNodeNames(node.getRight());
		String leftName = (node.getLeft().isLeaf()) ? node.getLeft().getOrganismName() : node.getLeft().getOrganismName().substring(0, node.getLeft().getOrganismName().indexOf("-"));
		String rightName = (node.getRight().isLeaf()) ? node.getRight().getOrganismName() : node.getRight().getOrganismName().substring(node.getRight().getOrganismName().indexOf("-")+1);
		node.setOrganismName(leftName + "-" + rightName);
	}

	/**
	 * Loads the genomes.
	 * @param genomeData
	 * @throws IOException
	 * @see Genome
	 * @see PhylogenyNode
	 * @see Phylogeny#genes
	 */
	public void loadGenomes(Reader genomeData) throws IOException {
		BufferedReader genomeDataBufferedReader = new BufferedReader(genomeData);
		Map<String, List<String>> organismGenomeStrings = new HashMap<String, List<String>>();
		String line = null;
		while ((line = genomeDataBufferedReader.readLine()) != null) {
			String organismName = line.substring(0, line.indexOf(" "));
			String genomeString = line.substring(line.indexOf(" ")+1);
			List<String> genomes = organismGenomeStrings.get(organismName);
			if (genomes==null) {
				genomes = new ArrayList<String>();
				organismGenomeStrings.put(organismName, genomes);
			}
			genomes.add(genomeString);
		}
		for (PhylogenyNode leaf : this.leaves) {
			List<String> genomeStrings = organismGenomeStrings.get(leaf.getOrganismName());
			if (genomeStrings!=null) {
				for (String genomeString : genomeStrings) {
					leaf.addCandidate(new Genome(genomeString));
				}
			}
		}
		
		loadGeneCount();
	}
	
	private void loadGeneCount() {
		this.geneCount = 0;
		for (PhylogenyNode leaf : this.leaves) {
			for (Genome g : leaf.getCandidates()) {
				if (this.geneCount==0) this.geneCount = g.getGeneCount();
				else if (this.geneCount!=g.getGeneCount()) throw new RuntimeException("not equal gene count");
			}
		}
	}
	
	@Override
	public String toString() {
		return root.toString();
	}
	
	public String getResult(EvolutionModel model) {
		return root.getResult(model);
	}
	
	/**
	 * Returns Phylogeny score. Does not include chromosome penalty.
	 * @param model
	 * @return score
	 */
	public int getScore(EvolutionModel model) {
		return phylogenyScore(this.getRoot(), model);
	}
	
	private static int phylogenyScore(PhylogenyNode node, EvolutionModel model) {
		if (node.isLeaf()) return 0;
		int leftDistance = model.distance(node.getGenome(), node.getLeft().getGenome());
		int rightDistance = model.distance(node.getGenome(), node.getRight().getGenome());
				
		return leftDistance + rightDistance +
				phylogenyScore(node.getLeft(), model) +
				phylogenyScore(node.getRight(), model);
	}

	public String getTreeStructure() {
		return getTreeStructure(getRoot());
	}
	
	private String getTreeStructure(PhylogenyNode node) {
		if (node.isLeaf())
			return node.getOrganismName();
		else {
			return "(" + getTreeStructure(node.getLeft()) + "," + getTreeStructure(node.getRight()) + ")";
		}
	}
}
