import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * A node of the phylogenetic tree.
 * @see Phylogeny
 */
public class PhylogenyNode {
	
	private String organismName;
	
	private PhylogenyNode left;
	private PhylogenyNode right;
	private PhylogenyNode parent;
	private boolean leaf;
	
	/**
	 * Genome of the organism.
	 */
	private Genome genome = null;

	
	/**
	 * Array of candidates
	 */
	private List<Genome> candidates = new ArrayList<Genome>();
	private Set<Genome> candidateSet = new HashSet<>();
	
	private List<int[]> candidateDiffBreak = new ArrayList<>();
	private List<int[]> candidateDiffJoin = new ArrayList<>();
	
	/**
	 * Tabu penalty of a candidate
	 */
	private List<Float> candidatePenalty = new ArrayList<>();
	
	/**
	 * Score of a candidate for the left sub-tree
	 */
	private List<Float> candidateScoreLeft = new ArrayList<>();
	
	/**
	 * Score of a candidate for the right sub-tree
	 */
	private List<Float> candidateScoreRight = new ArrayList<>();
	
	/**
	 * solutionsLeft[i] contains the optimal candidates in the left child, if i-th candidate is selected in the current node 
	 */
	private List<List<Integer>> solutionsLeft = new ArrayList<>();
	
	/**
	 * solutionsRight[i] contains the optimal candidates in the right child, if i-th candidate is selected in the current node 
	 */
	private List<List<Integer>> solutionsRight = new ArrayList<>();
	
	/**
	 * How many solutions exists with opt. score for the candidate
	 */
	private List<Integer> solutionCount = new ArrayList<>();
	
	public String getOrganismName() {
		return organismName;
	}
	
	public void setOrganismName(String organismName) {
		this.organismName = organismName;
	}
	
	public PhylogenyNode getLeft() {
		return left;
	}
	
	public void setLeft(PhylogenyNode left) {
		this.left = left;
	}
	
	public PhylogenyNode getRight() {
		return right;
	}
	
	public void setRight(PhylogenyNode right) {
		this.right = right;
	}
	
	public PhylogenyNode getParent() {
		return parent;
	}
	
	public void setParent(PhylogenyNode parent) {
		this.parent = parent;
	}
	
	public boolean isLeaf() {
		return leaf;
	}
	
	public void setLeaf(boolean leaf) {
		this.leaf = leaf;
	}
	
	public void setGenome(Genome genome) {
		this.genome = genome;
	}
	
//	public void addGenome(Genome genome) {
//		this.genomes.add(genome);
//		if (isLeaf()) this.solutionCountLeaf.add(1);
//	}
	
	public Genome getGenome() {
		return genome;
	}
	
	public List<Genome> getCandidates() {
		return candidates;
	}
	
	public int getCandidateSize() {
		return candidates.size();
	}
	
	public Genome getCandidate(int index) {
		return candidates.get(index);
	}
	
	public void clearCandidates() {
		this.candidates.clear();
		this.candidateSet.clear();
		this.candidateDiffBreak.clear();
		this.candidateDiffJoin.clear();
		this.candidatePenalty.clear();
		this.candidateScoreLeft.clear();
		this.candidateScoreRight.clear();
		this.solutionsLeft.clear();
		this.solutionsRight.clear();
		this.solutionCount.clear();
	}
	
	public boolean addCandidate(Genome candidate) {
		if (this.candidateSet.contains(candidate)) return false;
		else {
			this.candidates.add(candidate);
			this.candidateSet.add(candidate);
			this.candidateDiffBreak.add(null);
			this.candidateDiffJoin.add(null);
			this.candidatePenalty.add(0f);
			this.candidateScoreLeft.add(-1f);
			this.candidateScoreRight.add(-1f);
			this.solutionsLeft.add(new ArrayList<Integer>());
			this.solutionsRight.add(new ArrayList<Integer>());
			if (isLeaf()) this.solutionCount.add(1);
			else this.solutionCount.add(0);
			return true;
		}
	}
	
	public boolean addCandidate(Genome candidate, int[] diffBreak, int[] diffJoin) {
		if (this.addCandidate(candidate)) {
			int lastIndex = getCandidateSize()-1;
			this.candidateDiffBreak.set(lastIndex, diffBreak);
			this.candidateDiffJoin.set(lastIndex, diffJoin);
			return true;
		}
		return false;
	}
	
	public boolean isCandidateDiff(int index) {
		if (getCandidateDiffBreak(index)!=null && getCandidateDiffJoin(index)!=null) return true;
		return false;
	}
	
	public int[] getCandidateDiffBreak(int index) {
		return candidateDiffBreak.get(index);
	}
	
	public int[] getCandidateDiffJoin(int index) {
		return candidateDiffJoin.get(index);
	}
	
	public void addCandidatePenalty(int index, float penalty) {
		float p = this.candidatePenalty.get(index);
		this.candidatePenalty.set(index, p+penalty);
	}
	
	public float getCandidateScore(int candidate) {
		if (isLeaf()) return 0;
		return candidateScoreLeft.get(candidate)+candidateScoreRight.get(candidate)+ candidatePenalty.get(candidate);
	}
	
	public void setCandidateScoreLeft(int index, float score) {
		this.candidateScoreLeft.set(index, score);
	}
	
	public void setCandidateScoreRight(int index, float score) {
		this.candidateScoreRight.set(index, score);
	}
	
	public List<Integer> getLeftSolutions(int index) {
		return this.solutionsLeft.get(index);
	}
	
	public void addLeftSolution(int index, int cand) {
		this.solutionsLeft.get(index).add(cand);
	}
	
	public void deleteLeftSolutions(int index) {
		this.solutionsLeft.get(index).clear();
	}
	
	public List<Integer> getRightSolutions(int index) {
		return this.solutionsRight.get(index);
	}
	
	public void addRightSolution(int index, int cand) {
		this.solutionsRight.get(index).add(cand);
	}
	
	public void deleteRightSolutions(int index) {
		this.solutionsRight.get(index).clear();
	}
	
	public int getSolutionCount(int index) {
		if (isLeaf()) return 1;
		return solutionCount.get(index);
	}
	
	public List<Integer> getSolutionCounts() {
		return solutionCount;
	}
	
	public void setSolutionCount(int index, int count) {
		this.solutionCount.set(index, count);
	}
	
	public PhylogenyNode() {
		this.organismName = null;
		this.parent = null;
		this.left = null;
		this.right = null;
		this.leaf = false;
	}
	
	@Override
	public String toString() {
		StringBuilder output = new StringBuilder();
		if (leaf) {
			output.append(organismName);
		}
		else {
			output.append("(").append(left).append(",").append(right).append(")");
		}
		return output.toString();
	}

	public String getResult(EvolutionModel model) {
		String result = "";
		
		if (!isLeaf()) {
			result += this.getLeft().getResult(model);
			result += this.getRight().getResult(model);
		}

		int distToParent = 0;
		if (this.getParent()!=null) {
			distToParent = model.distance(this.getGenome(), this.getParent().getGenome());
		}
		return result + organismName + " " + distToParent + " " + this.getGenome().getStringRepresentation() + "\n";
	}
}
